% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tsl_transform.R
\name{tsl_transform}
\alias{tsl_transform}
\title{Transform Values in Time Series Lists}
\usage{
tsl_transform(tsl = NULL, f = NULL, ...)
}
\arguments{
\item{tsl}{(required, list) Time series list. Default: NULL}

\item{f}{(required, transformation function) name of a function taking a matrix as input. Currently, the following options are implemented, but any other function taking a matrix as input (for example, \code{\link[=scale]{scale()}}) should work as well:
\itemize{
\item f_proportion: proportion computed by row.
\item f_percent: percentage computed by row.
\item f_hellinger: Hellinger transformation computed by row
\item f_scale_local: Local centering and/or scaling based on the variable mean and standard deviation in each time series within \code{tsl}.
\item f_scale_global: Global centering and/or scaling based on the variable mean and standard deviation across all time series within \code{tsl}.
\item f_smooth: Time series smoothing with a user defined rolling window.
\item f_detrend_difference: Differencing detrending of time series via \code{\link[=diff]{diff()}}.
\item f_detrend_linear: Detrending of seasonal time series via linear modeling.
\item f_detrend_gam: Detrending of seasonal time series via Generalized Additive Models.
}}

\item{...}{(optional, additional arguments of \code{f}) Optional arguments for the transformation function.}
}
\value{
time series list
}
\description{
Function for time series transformations without changes in data dimensions. Generally, functions introduced via the argument \code{f} should not change the dimensions of the output time series list. See \code{\link[=tsl_resample]{tsl_resample()}} and \code{\link[=tsl_aggregate]{tsl_aggregate()}} for transformations requiring changes in time series dimensions.

This function supports a parallelization setup via \code{\link[future:plan]{future::plan()}}, and progress bars provided by the package \href{https://CRAN.R-project.org/package=progressr}{progressr}.
}
\examples{

#two time series
tsl <- tsl_initialize(
  x = fagus_dynamics,
  name_column = "name",
  time_column = "time"
) |>
  tsl_subset(
    names = c("Spain", "Sweden"),
    colnames = c("rainfall", "temperature")
  )

if(interactive()){
  tsl_plot(
    tsl = tsl
  )
}

#centering and scaling
#-----------------------------------------
#same mean and standard deviation are used to scale each variable across all time series
tsl_scale <- tsl_transform(
  tsl = tsl,
  f = f_scale_local
)

if(interactive()){
  tsl_plot(
    tsl = tsl_scale,
    guide_columns = 3
  )
}


#rescaling to a new range
#-----------------------------------------

#rescale between -100 and 100
tsl_rescaled <- tsl_transform(
  tsl = tsl,
  f = f_rescale_local,
  new_min = -100,
  new_max = 100
)

#old range
sapply(X = tsl, FUN = range)

#new range
sapply(X = tsl_rescaled, FUN = range)



#numeric transformations
#-----------------------------------------
#eemian pollen counts
tsl <- tsl_initialize(
  x = distantia::eemian_pollen,
  name_column = "name",
  time_column = "time"
)

if(interactive()){
  tsl_plot(
    tsl = tsl
  )
}

#percentages
tsl_percentage <- tsl_transform(
  tsl = tsl,
  f = f_percent
)

if(interactive()){
  tsl_plot(
    tsl = tsl_percentage
  )
}

#hellinger transformation
tsl_hellinger <- tsl_transform(
  tsl = tsl,
  f = f_hellinger
)

if(interactive()){
  tsl_plot(
    tsl = tsl_hellinger
  )
}

}
\seealso{
Other tsl_processing: 
\code{\link{tsl_aggregate}()},
\code{\link{tsl_resample}()},
\code{\link{tsl_smooth}()},
\code{\link{tsl_stats}()}
}
\concept{tsl_processing}
