% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{depth.curve.Tukey}
\alias{depth.curve.Tukey}
\title{Calculate Tukey curve depth using given points}
\usage{
depth.curve.Tukey(objects, data, nDirs = 100L, subs = TRUE, fracInt = 0.5,
  fracEst = 0.5, subsamples = NULL, exactEst = TRUE, minMassObj = 0,
  minMassDat = 0)
}
\arguments{
\item{objects}{A list where each element is a multivariate curve being a
list containing a matrix \code{coords} (values, d columns).}

\item{data}{A list where each element is a multivariate curve being a list
containing a matrix \code{coords} (values, d columns). The depths are
computed w.r.t. this data set.}

\item{nDirs}{Number of directions used to inspect the space, drawn from the
uniform distribution on the sphere.}

\item{subs}{Whether to split each object into two disjunctive subsets (one
for integrating and one for estimation) when computing the depth.}

\item{fracInt}{Portion of an object used for integrating.}

\item{fracEst}{Portion of an object used for estimation,
maximum: \code{1 - fracInt}.}

\item{subsamples}{A list indicating subsamples of points for each curve in
\code{objects}. Each elemnt of the list corresponds to a single curve and
should be given as a vector of the length equal to the number of points
on it, with entries indicating:
\itemize{
  \item 0 do not take the point into account at all,
  \item 1 use point as a reference (i.e. for integrating) and thus
         calculate its depth,
  \item 2 utilize point in depth calculation (i.e. for estimation).}}

\item{exactEst}{Is calculation of depth for each reference point of the
curve exact (\code{TRUE}, by default) or approximate (\code{FALSE}).}

\item{minMassObj}{Minimal portion of the \code{objects} distribution in the
halfspace to be considered when calculating depth.}

\item{minMassDat}{minimal portion of the \code{data} distribution in the
halfspace to be considered when calculating depth.}
}
\value{
A vector of doubles having the same length as \code{objects}, whose
each entry is the depth of each element of \code{objects} w.r.t.
\code{data}.
}
\description{
Calculates Tukey curve depth of each curve in \code{objects} w.r.t. the
sample of curves in \code{data}. Calculation of partial depth of each
single point can be either exact or approximate. If exact, modified method
of Dyckerhoff and Mozharovskyi (2016) is used; if approximate,
approximation is performed by projections on directions - points uniformly
distributed on the unit hypersphere.
}
\examples{
library(curveDepth)
# Load digits and transform them to curves
data("mnistShort017")
n <- 10 # cardinality of each class
m <- 50 # number of points to sample
cst <- 1/10 # a threshold constant
alp <- 1/8 # a threshold constant
curves0 <- images2curves(mnistShort017$`0`[, , 1:n])
curves1 <- images2curves(mnistShort017$`1`[, , 1:n])
set.seed(1)
curves0Smpl <- sample.curves(curves0, 2 * m)
curves1Smpl <- sample.curves(curves1, 2 * m)
# Calculate depths
depthSpace = matrix(NA, nrow = n * 2, ncol = 2)
depthSpace[, 1] = depth.curve.Tukey(
  c(curves0Smpl, curves1Smpl), curves0Smpl,
  exactEst = TRUE, minMassObj = cst/m^alp)
depthSpace[, 2] = depth.curve.Tukey(
  c(curves0Smpl, curves1Smpl), curves1Smpl,
  exactEst = TRUE, minMassObj = cst/m^alp)
# Draw the DD-plot
plot(NULL, xlim = c(0, 1), ylim = c(0, 1),
     xlab = paste("Depth w.r.t. '0'"),
     ylab = paste("Depth w.r.t. '1'"),
     main = paste("DD-plot for '0' vs '1'"))
grid()
# Draw the separating rule
dat1 <- data.frame(cbind(
  depthSpace, c(rep(0, n), rep(1, n))))
ddalpha1 <- ddalpha.train(X3 ~ X1 + X2, data = dat1,
                          depth = "ddplot",
                          separator = "alpha")
ddnormal <- ddalpha1$classifiers[[1]]$hyperplane[2:3]
pts <- matrix(c(0, 0, 1, ddnormal[1] / -ddnormal[2]),
              nrow = 2, byrow = TRUE)
lines(pts, lwd = 2)
# Draw the points
points(depthSpace[1:n, ],
       col = "red", lwd = 2, pch = 1)
points(depthSpace[(n + 1):(2 * n), ],
       col = "blue", lwd = 2, pch = 3)
}
\references{
Lafaye De Micheaux, P., Mozharovskyi, P. and Vimond, M. (2018).
Depth for curve data and applications.

Dyckerhoff, R. and Mozharovskyi P. (2016). Exact computation of the
halfspace depth. \emph{Computational Statistics and Data Analysis}, 98,
19-30.
}
