#' Plot the simulated baseline functions and histograms of simulated data
#'
#' This function takes the output of \code{\link[coxed]{sim.survdata}} and plots the baseline
#' failure PDF, the baseline failure CDF, the baseline survivor function, and the baseline hazard
#' function that were generated and used to simulate the data.  The function also produces histograms
#' of the simulated durations, the linear predictor, and the exponentiated linear predictor.
#' @param survsim An object of class "\code{simSurvdata}" output by \code{\link[coxed]{sim.survdata}}
#' @param type If \code{type="baseline"} the function plots the baseline functions.
#' If \code{type="hist"} the function plots the histograms. If \code{type="both"} the function
#' plots both the baseline functions and the histograms, aligned in a 2 x 1 array
#' @param bins The number of bins to draw in each histogram. Ignored if \code{type="baseline"}
#' @param df If \code{survsim} is generated by a call to \code{\link[coxed]{sim.survdata}} in which
#' \code{num.data.frames} is greater than 1, the output is a list of iterated simulation output. In this case,
#' \code{df} specifies the particular simulation iteration for which to produce visualizations
#' @details A challenge that can limit research to develop methods to analyze survival data is that a
#' pre-specified baseline hazard function must be used to generate simulated data, which contradicts the
#' Cox proportional hazards model's feature of circumventing the hazard to estimate coefficients.  The
#' flexible-hazard method developed by Harden and Kropko (2018) and implemented in \code{\link[coxed]{sim.survdata}}
#' allows for the simulation of duration data without assuming a particular parametric form for hazard.
#' \code{survsim.plot} is useful for visualizing the baseline functions, including hazard, that result from
#' this algorithm for a particular draw of simulated duration data. The function uses \code{\link[ggplot2]{ggplot}}
#' to create line plots for the baseline failure PDF, the baseline failure CDF, the baseline survivor function,
#' and the baseline hazard function over time.  The baseline functions and time are attributes of the
#' \code{\link[coxed]{sim.survdata}} output.
#' @return A figure of class \code{"gg"} and \code{"ggplot"} if \code{type="baseline"} or \code{type="hist"}, and of
#' classes \code{"gtable"}, \code{"gTree"}, \code{"grob"}, and \code{"gDesc"} if \code{type="both"}.
#' @references Harden, J. J. and Kropko, J. (2018). Simulating Duration Data for the Cox Model.
#' \emph{Political Science Research and Methods} \url{https://doi.org/10.1017/psrm.2018.19}
#' @author Jonathan Kropko <jkropko@@virginia.edu> and Jeffrey J. Harden <jharden2@@nd.edu>
#' @examples
#' simdata <- sim.survdata(N=1000, T=100, num.data.frames=1)
#' survsim.plot(simdata)
#' @export
#' @seealso \code{\link[coxed]{sim.survdata}}, \code{\link[ggplot2]{ggplot}}, \code{\link[gridExtra]{grid.arrange}}
survsim.plot <- function(survsim, type="both", bins=30, df=1){
     stopifnot(inherits(survsim, "simSurvdata"))
     if(!is.null(type) & !(type %in% c("baseline", "hist", "both")))
          stop("type must be one of baseline, hist, or both")
     if(inherits(survsim, "simSurvdataList")) survsim <- survsim[[df]]
     if(type=="baseline") g <- baseline.plot(survsim$baseline)
     if(type=="hist") g <- data.plot(survsim$data, survsim$xb, bins=bins)
     if(type=="both"){
          g1 <- baseline.plot(survsim$baseline)
          g2 <- data.plot(survsim$data, survsim$xb, bins=bins)
          g <- gridExtra::grid.arrange(g1, g2, nrow=2)
     }
     return(g)
}
