\name{colorhcplot}
\alias{colorhcplot}
\title{
Colorful Hierarchical Clustering Dendrograms
}
\description{
This function takes a "hclust-class" object and a factor describing the groups as arguments and generates colorful dendrograms in which leaves belonging to different groups are identified by colors. This function produces a plot that allows to easily detect if leaves that clustered together also belong to the same group.
}
\usage{
colorhcplot(hc, fac, hang = 0.1, 
            main = "Cluster Dendrogram", 
            colors = NULL, lab.cex = 1, 
            ylim = NULL, lwd = 3, 
            las = 1, lab.mar = 0.55)
}
\arguments{
  \item{hc}{hclust-class object, typically the result of a 'hclust()' function call.}
  \item{fac}{factor that defines the grouping.}
  \item{hang}{hang value, as in \code{hclust}. hang is the fraction of the plot height by which labels should hang below the rest of the plot. A negative value will align all labels at the bottom of the plot.}
  \item{main}{
title of the dendrogram plot.
}
  \item{colors}{
NULL or a character vector of length 1 or having the same length as the number of levels in fac. This argument defines the palette for the plot.}
  \item{lab.cex}{
numeric value for adjusting the font size of the leaf labels (and legend text).
}
  \item{ylim}{
numeric, defines the minimum and maximum value of the y-axis of the plot.
}
  \item{lwd}{
numeric value that defines the width (in points) of the lines of the dendogram.
}
  \item{las}{
graphic parameter for the orientation of the y-axis tick labels.
}
  \item{lab.mar}{
fraction of the plot area that is reserved for the labels (at the bottom of the plot).
}
}
\details{
In order to generate a colorful dendrogram, the colorhcplot() function requires 2 mandatory arguments: hc and fac. hc is the result of a hclust() call, while fac is a factor defining the groups. The number of leaves of the dendrogram has to be identical to the length of fac.
}
\value{
Calling colorhcplot() returns a colorful dendrogram plot}
\author{
Damiano Fantini <damiano.fantini@gmail.com>
}
\note{
Online colorhcplot() function reference at: \url{http://www.biotechworld.it/bioinf/2015/09/30/colorful-hierarchical-clustering-dendrograms-with-r}
}
\seealso{
\link{hclust}
}
\examples{
### Example 1, using the USArrests dataset
data(USArrests)
hc <- hclust(dist(USArrests), "ave")
fac <- as.factor(c(rep("group 1", 10), 
                   rep("group 2", 10), 
                   rep("unknown", 30)))
plot(hc)
colorhcplot(hc, fac)
colorhcplot(hc, fac, hang = -1, lab.cex = 0.8)

### Example 2: use the "ward.D2" algorithm and
### the UScitiesD dataset
data(UScitiesD)
hcity.D2 <- hclust(UScitiesD, "ward.D2")
fac.D2 <-as.factor(c(rep("group1", 3), 
                     rep("group2", 7)))
plot(hcity.D2, hang=-1)
colorhcplot(hcity.D2, fac.D2, color = c("chartreuse2", "orange2"))
colorhcplot(hcity.D2, fac.D2, color = "gray30", lab.cex = 1.2, lab.mar = 0.75)

### Example 3: use gene expression data 
data(geneData, package="colorhcplot")
exprs <- geneData$exprs
fac <- geneData$fac
hc <- hclust(dist(t(exprs)))
colorhcplot(hc, fac, main ="default", col = "gray10")
colorhcplot(hc, fac, main="Control vs. Tumor Samples") 
}
