% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/processing.R
\name{summarize_sedc}
\alias{summarize_sedc}
\title{Calculate Sum of Exponentially Decaying Contributions (SEDC) covariates}
\usage{
summarize_sedc(
  point_from = NULL,
  point_to = NULL,
  id = NULL,
  sedc_bandwidth = NULL,
  threshold = NULL,
  target_fields = NULL,
  extent_from = NULL,
  extent_to = NULL,
  ...
)
}
\arguments{
\item{point_from}{\code{SpatVector} object. Locations where
the sum of SEDCs are calculated.}

\item{point_to}{\code{SpatVector} object.
Locations where each SEDC is calculated.}

\item{id}{character(1). Name of the unique id field in \code{point_to}.}

\item{sedc_bandwidth}{numeric(1).
Distance at which the source concentration is reduced to
\code{exp(-3)} (approximately -95 \%)}

\item{threshold}{numeric(1). For computational efficiency,
the nearest points in threshold will be selected.
\code{2 * sedc_bandwidth} is applied if this value remains \code{NULL}.}

\item{target_fields}{character. Field names to calculate SEDC.}

\item{extent_from}{numeric(4) or SpatExtent. Extent of clipping \code{point_from}.
It only works with \code{point_from} of character(1) file path.
See \code{\link[terra:ext]{terra::ext}} for more details. Coordinate systems should match.}

\item{extent_to}{numeric(4) or SpatExtent. Extent of clipping \code{point_to}.}

\item{...}{Placeholder.}
}
\value{
data.frame object with input field names with
a suffix \code{"_sedc"} where the sums of EDC are stored.
Additional attributes are attached for the EDC information.
\itemize{
\item \code{attr(result, "sedc_bandwidth")}: the bandwidth where
concentration reduces to approximately five percent
\item \code{attr(result, "sedc_threshold")}: the threshold distance
at which emission source points are excluded beyond that
}
}
\description{
Calculate Sum of Exponentially Decaying Contributions (SEDC) covariates
}
\details{
The SEDC is specialized in vector to vector summary of covariates
with exponential decay. Decaying slope will be defined by \code{sedc_bandwidth},
where the concentration of the source is reduced to $\\exp(-3)$
(approximately 5 \\%). This function is useful when users a proper theory
of the attenuating concentration with the distance from the sources.
It can be thought of as a fixed bandwidth kernel weighted sum of covariates,
which encapsulates three steps:
\itemize{
\item Calculate the distance between each source and target points.
\item Calculate the weight of each source point with the exponential decay.
\item Summarize the weighted covariates.
}
}
\note{
Distance calculation is done with \code{terra} functions internally.
Thus, the function internally converts \code{sf} objects in
\verb{point_*} arguments to \code{terra}. Please note that any \code{NA} values
in the input will be ignored in SEDC calculation.
}
\examples{
library(terra)
library(sf)
set.seed(101)
ncpath <- system.file("gpkg/nc.gpkg", package = "sf")
nc <- terra::vect(ncpath)
nc <- terra::project(nc, "EPSG:5070")
pnt_from <- terra::centroids(nc, inside = TRUE)
pnt_from <- pnt_from[, "NAME"]
pnt_to <- terra::spatSample(nc, 100L)
pnt_to$pid <- seq(1, 100)
pnt_to <- pnt_to[, "pid"]
pnt_to$val1 <- rgamma(100L, 1, 0.05)
pnt_to$val2 <- rgamma(100L, 2, 1)

vals <- c("val1", "val2")
suppressWarnings(
  summarize_sedc(pnt_from, pnt_to, "NAME", 1e5, 2e5, vals)
)
}
\references{
\itemize{
\item Messier KP, Akita Y, Serre ML. (2012). Integrating Address Geocoding, Land Use Regression, and Spatiotemporal Geostatistical Estimation for Groundwater Tetrachloroethylene. \emph{Environmental Science & Technology} 46(5), 2772-2780.(\doi{10.1021/es203152a})
\item Wiesner C. (n.d.). Euclidean Sum of Exponentially Decaying Contributions Tutorial.
}
}
\seealso{
Other Macros for calculation: 
\code{\link{extract_at}()},
\code{\link{kernelfunction}()},
\code{\link{summarize_aw}()}
}
\author{
Insang Song
}
\concept{Macros for calculation}
