% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/analysis_random_graphs.R, R/random_graphs.R
\name{RandomGraphs}
\alias{RandomGraphs}
\alias{analysis_random_graphs}
\alias{sim.rand.graph.par}
\alias{sim.rand.graph.clust}
\title{Perform an analysis with random graphs for brain MRI data}
\usage{
analysis_random_graphs(g.list, N = 100, savedir = ".", ...)

sim.rand.graph.par(g, N = 100, clustering = FALSE, ...)

sim.rand.graph.clust(g, rewire.iters = 10000, cl = g$transitivity,
  max.iters = 100)
}
\arguments{
\item{g.list}{List of lists containing \code{igraph} graph objects}

\item{N}{Integer; the number of random graphs to simulate (default: 100)}

\item{savedir}{Character string specifying the directory in which to save the
generated graphs (default: current working directory)}

\item{...}{Other parameters (passed to \code{\link{sim.rand.graph.clust}})}

\item{g}{An \code{igraph} graph object}

\item{clustering}{Logical; whether or not to control for clustering (default:
\code{FALSE})}

\item{rewire.iters}{Integer; number of rewiring iterations for the initial
graph randomization (default: 1e4)}

\item{cl}{The clustering measure (default: transitivity)}

\item{max.iters}{The maximum number of iterations to perform; choosing a
lower number may result in clustering that is further away from the
observed graph's (default: 100)}
}
\value{
\code{analysis_random_graphs} returns a \emph{list} containing:
\item{rich}{A data table containing normalized rich-club coefficients and
  p-values}
\item{small}{A data table with small-world parameters}
\item{rand}{A data table with some global graph measures for all random
  graphs generated}

\code{sim.rand.graph.par} - a \emph{list} of \emph{N} random graphs
  with some additional vertex and graph attributes

\code{sim.rand.graph.clust} - A single \code{igraph} graph object
}
\description{
\code{analysis_random_graphs} is not quite a "proper" function. It performs
the steps needed for doing typical graph theory analyses with brain MRI data
if you need to generate equivalent random graphs. This includes calculating
\emph{small world} parameters and normalized \emph{rich club} coefficients.

\code{sim.rand.graph.par} simulates \code{N} simple random graphs with the
same clustering (optional) and degree sequence as the input. Essentially a
wrapper for \code{\link[igraph]{sample_degseq}} (or, if you want to match by
clustering, \code{\link{sim.rand.graph.clust}}) and
\code{\link{set_brainGraph_attr}}. It uses \code{\link[foreach]{foreach}} for
parallel processing.

\code{sim.rand.graph.clust} simulates a random graph with a given degree
sequence \emph{and} clustering coefficient. Increasing the \code{max.iters}
value will result in a closer match of clustering with the observed graph.
}
\details{
\code{analysis_random_graphs} does the following:
\enumerate{
  \item Generate \code{N} random graphs for each group and density/threshold
    (and subject if you have subject-specific graphs).
  \item Write graphs to disk in \code{savedir}. Read them back into \code{R}
    and combine into lists; then write these lists to disk (in a
    sub-directory named \code{ALL}), so you can delete the individual
    \code{.rds} files afterwards.
  \item Calculate \emph{small world} parameters, along with values for a few
    global graph measures that may be of interest.
  \item Calculate \emph{normalized rich club coefficients} and associated
    p-values.
}

If you do not want to match by clustering, then simple rewiring of the input
graph is performed (the number of rewire's equaling the larger of \code{1e4}
and \eqn{10 \times m}, where \eqn{m} is the graph's edge count).
}
\examples{
\dontrun{
rand_all <- analysis_random_graphs(g.norm, 1e2,
  savedir='/home/cwatson/dti/rand', clustering=F)
}
\dontrun{
rand1 <- sim.rand.graph.par(g[[1]][[N]], N=1e3)
rand1.cl <- sim.rand.graph.par(g[[1]][[N]], N=1e2,
  clustering=T, max.iters=1e3)
}
}
\references{
Bansal S., Khandelwal S., Meyers L.A. (2009) \emph{Exploring
biological network structure with clustered random networks}. BMC
Bioinformatics, 10:405-421.
}
\seealso{
\code{\link{small.world}}

\code{\link[igraph]{rewire}, \link[igraph]{sample_degseq},
\link[igraph]{keeping_degseq}}

\code{\link[igraph]{transitivity}}

Other Random graph functions: \code{\link{RichClub}}
}
\author{
Christopher G. Watson, \email{cgwatson@bu.edu}
}
\concept{Random graph functions}
