## for R-check
wishart <- "ignored";
blmer <- function(formula, data = NULL, REML = TRUE,
                  control = lmerControl(), start = NULL,
                  verbose = 0L, subset, weights, na.action, offset,
                  contrasts = NULL, devFunOnly = FALSE,
                  cov.prior = wishart, fixef.prior = NULL,
                  resid.prior = NULL,
                  ...)
{
  mc <- mcout <- match.call()
  missCtrl <- missing(control)
  missCovPrior <- missing(cov.prior)
  ## see functions in modular.R for the body ...
  if (!missCtrl && !inherits(control, "lmerControl")) {
    if(!is.list(control)) stop("'control' is not a list; use lmerControl()")
    ## back-compatibility kluge
    warning("passing control as list is deprecated: please use lmerControl() instead",
            immediate.=TRUE)
    control <- do.call(lmerControl, control)
  }
  if (!is.null(list(...)[["family"]])) {
    warning("calling lmer with 'family' is deprecated; please use glmer() instead")
    mc[[1]] <- quote(lme4::glmer)
    if(missCtrl) mc$control <- glmerControl()
    return(eval(mc, parent.frame(1L)))
  }

  fixef.prior <- mc$fixef.prior; ## for delayed evaluation
  cov.prior <- if (!missCovPrior) mc$cov.prior else formals(blmer)$cov.prior;
  resid.prior <- mc$resid.prior;
  if (!is.null(mc$var.prior)) {
    resid.prior <- parse(text = mc$var.prior)[[1]];
  }
  mc$fixef.prior <- NULL;
  mc$cov.prior <- NULL;
  mc$resid.prior <- NULL;
  mc$var.prior <- NULL;

  sigmaFixed <- !is.null(resid.prior) && (grepl("^\\W*point", resid.prior) || (is.call(resid.prior) && resid.prior[[1]] == "point"));
  if (sigmaFixed) {
    control$checkControl$check.nobs.vs.nlev <- "ignore";
    control$checkControl$check.nobs.vs.rankZ <- "ignore";
    control$checkControl$check.nobs.vs.nRE <- "ignore";
  }
  mc$control <- control ## update for  back-compatibility kluge
  
  ## https://github.com/lme4/lme4/issues/50
  ## parse data and formula
  mc[[1]] <- quote(lme4::lFormula)
  lmod <- eval(mc, parent.frame(1L));
  mcout$formula <- lmod$formula
  lmod$formula <- NULL

  lmerStart <- NULL;
  if (!is.null(start) && is.list(start) && length(start) > 1)
    lmerStart <- start$theta;
  
  devfun <- do.call(mkBlmerDevfun,
                    c(lmod, lmod$X, lmod$reTrms,
                      list(priors = list(covPriors = cov.prior, fixefPrior = fixef.prior, residPrior = resid.prior),
                           start = lmerStart, verbose = verbose, control = control)));

  if (devFunOnly) return(devfun);
  
  devFunEnv <- environment(devfun);
  start <- getStartingValues(start, devFunEnv);

  opt <- optimizeLmer(devfun,
                      optimizer=control$optimizer,
                      restart_edge=control$restart_edge,
                      control=control$optCtrl,
                      verbose=verbose,
                      start=start)
  result <- mkMerMod(devFunEnv, opt, lmod$reTrms, fr = lmod$fr, mcout) ## prepare output
  
  result <- repackageMerMod(result, opt, devFunEnv);

  return(result);
}

bglmer <- function(formula, data=NULL, family = gaussian,
                  control = glmerControl(), start = NULL, verbose = 0L, nAGQ = 1L,
                  subset, weights, na.action, offset,
                  contrasts = NULL, mustart, etastart, devFunOnly = FALSE,
                  cov.prior = wishart, fixef.prior = NULL,
                  ...)
{
  missCovPrior <- missing(cov.prior);
  if (!inherits(control, "glmerControl")) {
    if(!is.list(control)) stop("'control' is not a list; use glmerControl()")
    ## back-compatibility kluge
    msg <- "Use control=glmerControl(..) instead of passing a list"
    if(length(cl <- class(control))) msg <- paste(msg, "of class", dQuote(cl[1]))
    warning(msg, immediate.=TRUE)
    control <- do.call(glmerControl, control)
  }
  mc <- mcout <- match.call()
  
  fixef.prior <- mc$fixef.prior; ## for delayed evaluation
  cov.prior <- if (!missCovPrior) mc$cov.prior else formals(bglmer)$cov.prior;
  mc$fixef.prior <- NULL;
  mc$cov.prior <- NULL;
  
  ## family-checking code duplicated here and in glFormula (for now) since
  ## we really need to redirect at this point; eventually deprecate formally
  ## and clean up
  if (is.character(family))
    family <- get(family, mode = "function", envir = parent.frame(2))
  if( is.function(family)) family <- family()
  if (isTRUE(all.equal(family, gaussian()))) {
    ## redirect to lmer (with warning)
    warning("calling glmer() with family=gaussian (identity link) as a shortcut to lmer() is deprecated;",
            " please call lmer() directly")
    mc[[1]] <- quote(lme4::lmer)
    mc["family"] <- NULL            # to avoid an infinite loop
    return(eval(mc, parent.frame()))
  }
  
  ## see https://github.com/lme4/lme4/issues/50
  ## parse the formula and data
  mc[[1]] <- quote(lme4::glFormula)
  glmod <- eval(mc, parent.frame(1L))
  mcout$formula <- glmod$formula
  glmod$formula <- NULL
  
  ## create deviance function for covariance parameters (theta)
  
  devfun <- do.call(mkBglmerDevfun, c(glmod, glmod$X, glmod$reTrms,
                                      list(priors = list(covPriors = cov.prior, fixefPrior = fixef.prior),
                                           verbose=verbose, control=control, nAGQ = 0)))
  if (nAGQ==0 && devFunOnly) return(devfun)
  ## optimize deviance function over covariance parameters
  
  if (is.list(start) && !is.null(start$fixef))
    if (nAGQ==0) stop("should not specify both start$fixef and nAGQ==0")
  
  opt <- optimizeGlmer(devfun,
                       optimizer = control$optimizer[[1]],
                       restart_edge=control$restart_edge,
                       control = control$optCtrl,
                       start=start,
                       nAGQ = 0,
                       verbose=verbose)
  
  if(nAGQ > 0L) {
    
    start <- get("updateStart", asNamespace("lme4"))(start,theta=opt$par)
    
    ## update deviance function to include fixed effects as inputs
    devfun <- updateBglmerDevfun(devfun, glmod$reTrms, nAGQ = nAGQ)
    
    if (devFunOnly) return(devfun)
    ## reoptimize deviance function over covariance parameters and fixed effects
    opt <- optimizeGlmer(devfun,
                         optimizer = control$optimizer[[2]],
                         restart_edge=control$restart_edge,
                         control = control$optCtrl,
                         start=start,
                         nAGQ=nAGQ,
                         verbose = verbose,
                         stage=2)
  }
  ## prepare output
  result <- mkMerMod(environment(devfun), opt, glmod$reTrms, fr = glmod$fr, mcout)
  result <- repackageMerMod(result, opt, environment(devfun));

  return(result);
}

lmmObjective <- function(pp, resp, sigma, exponentialTerms, polynomialTerm, blmerControl) {
  sigma.sq <- sigma^2;

  result <- resp$objective(pp$ldL2(), pp$ldRX2(), pp$sqrL(1.0), sigma.sq);

  exponentialTerm <- 0;
  for (i in 1:length(exponentialTerms)) {
    power <- as.numeric(names(exponentialTerms)[[i]]);
    value <- exponentialTerms[[i]];
    if (!is.finite(value)) return(value);
    
    exponentialTerm <- exponentialTerm + value * sigma^power;
  }

  priorPenalty <- exponentialTerm + polynomialTerm + blmerControl$constant + blmerControl$df * log(sigma.sq);
  
  result <- result + priorPenalty;

  return(result);
}

repackageMerMod <- function(merMod, opt, devFunEnv) {
  isLMM <- is(merMod, "lmerMod");

  if (isLMM)
    expandPars(opt$par, devFunEnv$pars);

  blmerControl <- devFunEnv$blmerControl;
  priors <- devFunEnv$priors;
  
  beta <- if (isLMM) merMod@pp$beta(1.0) else merMod@beta;
  Lambda.ts <- getCovBlocks(merMod@pp$Lambdat, blmerControl$ranefStructure);
  exponentialTerms <- calculatePriorExponentialTerms(priors, beta, Lambda.ts);

  if (isLMM) {
    if (!is.null(exponentialTerms[["-2"]]))
      merMod@devcomp$cmp[["pwrss"]] <- merMod@devcomp$cmp[["pwrss"]] + as.numeric(exponentialTerms[["-2"]]);
  
    ## recover sigma
    sigmaOptimizationType <- blmerControl$sigmaOptimizationType;
    if (sigmaOptimizationType == SIGMA_OPTIM_POINT) {
      sigma <- priors$residPrior@value;
    } else if (sigmaOptimizationType != SIGMA_OPTIM_NUMERIC) {
      profileSigma <- getSigmaProfiler(priors, blmerControl);
      sigma <- profileSigma(merMod@pp, merMod@resp, exponentialTerms, blmerControl);
    }
    ## set sigma in final object
    numObs   <- merMod@devcomp$dims[["n"]];
    numFixef <- merMod@devcomp$dims[["p"]];
    if (merMod@devcomp$dims[["REML"]] > 0L) {
      merMod@devcomp$cmp[["sigmaREML"]] <- sigma;
      merMod@devcomp$cmp[["sigmaML"]] <- sigma * sqrt((numObs - numFixef) / numObs);
    } else {
      merMod@devcomp$cmp[["sigmaML"]] <- sigma;
      merMod@devcomp$cmp[["sigmaREML"]] <- sigma * sqrt(numObs / (numObs - numFixef));
    }
    
    objectiveValue <- merMod@resp$objective(merMod@pp$ldL2(), merMod@pp$ldRX2(), merMod@pp$sqrL(1.0), sigma^2);
    if (merMod@devcomp$dims[["REML"]] > 0L) {
      priorPenalty <- merMod@devcomp$cmp[["REML"]] - objectiveValue;
      merMod@devcomp$cmp[["REML"]] <- objectiveValue;
    } else {
      priorPenalty <- merMod@devcomp$cmp[["dev"]] - objectiveValue;
      merMod@devcomp$cmp[["dev"]] <- objectiveValue;
    }
    merMod@devcomp$cmp[["penalty"]] <- priorPenalty;

    return(new("blmerMod",
               resp    = merMod@resp,
               Gp      = merMod@Gp,
               call    = merMod@call,
               frame   = merMod@frame,
               flist   = merMod@flist,
               cnms    = merMod@cnms,
               lower   = merMod@lower,
               theta   = merMod@theta,
               beta    = merMod@beta,
               u       = merMod@u,
               devcomp = merMod@devcomp,
               pp      = merMod@pp,
               optinfo = merMod@optinfo,
               priors  = priors));
  } else {
    if (length(exponentialTerms) > 0)
      priorPenalty <- exponentialTerms[[1]] + calculatePriorPolynomialTerm(priors$covPriors, Lambda.ts) + blmerControl$constant
    else
      priorPenalty <- 0;
    merMod@devcomp$cmp[["dev"]] <- merMod@devcomp$cmp[["dev"]] - priorPenalty;
    merMod@devcomp$cmp[["penalty"]] <- priorPenalty;

    return(new("bglmerMod",
               resp    = merMod@resp,
               Gp      = merMod@Gp,
               call    = merMod@call,
               frame   = merMod@frame,
               flist   = merMod@flist,
               cnms    = merMod@cnms,
               lower   = merMod@lower,
               theta   = merMod@theta,
               beta    = merMod@beta,
               u       = merMod@u,
               devcomp = merMod@devcomp,
               pp      = merMod@pp,
               optinfo = merMod@optinfo,
               priors  = priors));
  }
}

validateRegressionArgument <- function(regression, regressionName) {
  if (missing(regression)) stop("'regression' missing.");
  
  # check for existence and null-ness
  if (is.null(regression)) stop("object '", regressionName, "' is null.");
  if (!is(regression, "bmerMod")) stop("object '", regressionName, "' does not inherit from S4 class 'bmerMod'.");
}

setPrior <- function(regression, cov.prior = NULL,
                     fixef.prior = NULL, resid.prior = NULL, envir = parent.frame(1L), ...)
{
  matchedCall <- match.call();

  covMissing   <- missing(cov.prior);
  fixefMissing <- missing(fixef.prior);
  residMissing <- missing(resid.prior);
  
  validateRegressionArgument(regression, matchedCall$regression);
  
  if (residMissing && !is.null(matchedCall$var.prior)) {
    matchedCall$resid.prior <- matchedCall$var.prior;
    residMissing <- FALSE;
  }

  priors <- evaluatePriorArguments(matchedCall$cov.prior, matchedCall$fixef.prior, matchedCall$resid.prior,
                                   regression@devcomp$dim, regression@cnms, envir);

  if (!covMissing) regression@covPriors <- priors$covPriors;
  if (!fixefMissing) regression@fixefPrior <- priors$fixefPrior;
  if (!residMissing) regression@residPrior <- priors$residPrior;
  
  return (regression);
}

parsePrior <- function(regression, cov.prior = NULL,
                       fixef.prior = NULL, resid.prior = NULL, envir = parent.frame(), ...)
{
  matchedCall <- match.call();

  covMissing   <- missing(cov.prior);
  fixefMissing <- missing(fixef.prior);
  residMissing <- missing(resid.prior);
  
  validateRegressionArgument(regression, matchedCall$regression);
  
  if (residMissing && !is.null(matchedCall$var.prior)) {
    matchedCall$resid.prior <- matchedCall$var.prior;
    residMissing <- FALSE;
  }

  priors <- evaluatePriorArguments(matchedCall$cov.prior, matchedCall$fixef.prior, matchedCall$resid.prior,
                                   regression@devcomp$dim, regression@cnms, envir);

  result <- list();
  if (!covMissing) result$covPriors <- priors$covPriors;
  if (!fixefMissing) result$fixefPrior <- priors$fixefPrior;
  if (!residMissing) result$residPrior <- priors$residPrior;

  if (length(result) == 1) return(result[[1]]);
  return(result);
}

if (FALSE) {
runOptimizer <- function(regression, verbose = FALSE)
{
  validateRegressionArgument(regression, match.call()$regression);
  
  if (verbose) {
    regression@dims[["verb"]] <- as.integer(1)
  } else {
    regression@dims[["verb"]] <- as.integer(0)
  }
  return (mer_finalize(regression));
}

runOptimizerWithPrior <- function(regression, cov.prior = NULL,
                                  fixef.prior = NULL, var.prior = NULL,
                                  verbose = FALSE, envir = parent.frame())
{
  validateRegressionArgument(regression, match.call()$regression);
  
  regression <- setPrior(regression, cov.prior, fixef.prior, var.prior, envir);
  
  return(runOptimizer(regression, verbose));
}
}
