% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BIOMOD_RangeSize.R
\name{BIOMOD_RangeSize}
\alias{BIOMOD_RangeSize}
\alias{BIOMOD_RangeSize,data.frame,data.frame-method}
\alias{BIOMOD_RangeSize,array,array-method}
\alias{BIOMOD_RangeSize,RasterStack,RasterStack-method}
\alias{BIOMOD_RangeSize,RasterLayer,RasterStack-method}
\alias{BIOMOD_RangeSize,RasterLayer,RasterLayer-method}
\title{Analyze the range size differences between projections of species distribution models}
\usage{
BIOMOD_RangeSize(proj.current, proj.future)

\S4method{BIOMOD_RangeSize}{data.frame,data.frame}(proj.current, proj.future)

\S4method{BIOMOD_RangeSize}{array,array}(proj.current, proj.future)

\S4method{BIOMOD_RangeSize}{RasterStack,RasterStack}(proj.current, proj.future)

\S4method{BIOMOD_RangeSize}{RasterLayer,RasterStack}(proj.current, proj.future)

\S4method{BIOMOD_RangeSize}{RasterLayer,RasterLayer}(proj.current, proj.future)
}
\arguments{
\item{proj.current}{an \code{array}, \code{data.frame}, \code{\link[raster:stack]{RasterLayer}} 
or \code{\link[raster:stack]{RasterStack}} object containing the initial binary projection(s) 
of the (ensemble) species distribution model(s)}

\item{proj.future}{an \code{array}, \code{data.frame}, \code{\link[raster:stack]{RasterLayer}} 
or \code{\link[raster:stack]{RasterStack}} object containing the final binary projection(s) 
of the (ensemble) species distribution model(s)}
}
\value{
A \code{list} containing two objects :
\describe{
  \item{Compt.By.Species}{a \code{data.frame} containing the summary of range change for each 
  comparison
  \itemize{
    \item{\code{Loss} : }{number of pixels predicted to be lost}
    \item{\code{Stable0} : }{number of pixels not currently occupied and not predicted to be}
    \item{\code{Stable1} : }{number of pixels currently occupied and predicted to remain 
    occupied}
    \item{\code{Gain} : }{number of pixels predicted to be gained}
    \item{\code{PercLoss} : }{percentage of pixels currently occupied and predicted to be lost 
    (\code{Loss / (Loss + Stable1)})}
    \item{\code{PercGain} : }{percentage of pixels predicted to be gained compare to the 
    number of pixels currently occupied (\code{Gain / (Loss + Stable1)})}
    \item{\code{SpeciesRangeChange} : }{percentage of pixels predicted to change (loss or gain) 
    compare to the number of pixels currently occupied (\code{PercGain - PercLoss})}
    \item{\code{CurrentRangeSize} : }{number of pixels currently occupied}
    \item{\code{FutureRangeSize0Disp} : }{number of pixels predicted to be occupied, assuming 
    no migration}
    \item{\code{FutureRangeSize1Disp} : }{number of pixels predicted to be occupied, assuming 
    migration}
  }
  }
  \item{Diff.By.Pixel}{an object in the same form than the input data (\code{proj.current} and 
  \code{proj.future}) and containing a value for each point/pixel of each comparison among :
  \itemize{
    \item \code{-2} : predicted to be lost
    \item \code{-1} : predicted to remain occupied
    \item \code{0} : predicted to remain unoccupied
    \item \code{1} : predicted to be gained
  }
  }
}
}
\description{
This function allows to calculate the absolute number of locations (pixels) lost, 
stable and gained, as well as the corresponding relative proportions, between two (or more) 
binary projections of (ensemble) species distribution models (\emph{which can represent new 
time scales or environmental scenarios for example}).
}
\details{
Note that \bold{this function is only relevant to compare binary projections, made on the 
same area with the same resolution}.

\code{Diff.By.Pixel} object is obtained by applying the simple following formula :
\deqn{proj.future - 2 * proj.current}
}
\examples{

# Load species occurrences (6 species available)
myFile <- system.file('external/species/mammals_table.csv', package = 'biomod2')
DataSpecies <- read.csv(myFile, row.names = 1)
head(DataSpecies)

# Select the name of the studied species
myRespName <- 'GuloGulo'

# Get corresponding presence/absence data
myResp <- as.numeric(DataSpecies[, myRespName])

# Get corresponding XY coordinates
myRespXY <- DataSpecies[, c('X_WGS84', 'Y_WGS84')]

# Load environmental variables extracted from BIOCLIM (bio_3, bio_4, bio_7, bio_11 & bio_12)
myFiles <- paste0('external/bioclim/current/bio', c(3, 4, 7, 11, 12), '.grd')
myExpl <- raster::stack(system.file(myFiles, package = 'biomod2'))

\dontshow{
myExtent <- raster::extent(0,30,45,70)
myExpl <- raster::stack(raster::crop(myExpl, myExtent))
}

# ---------------------------------------------------------------
file.out <- paste0(myRespName, "/", myRespName, ".AllModels.models.out")
if (file.exists(file.out)) {
  myBiomodModelOut <- get(load(file.out))
} else {

  # Format Data with true absences
  myBiomodData <- BIOMOD_FormatingData(resp.var = myResp,
                                       expl.var = myExpl,
                                       resp.xy = myRespXY,
                                       resp.name = myRespName)

  # Create default modeling options
  myBiomodOptions <- BIOMOD_ModelingOptions()

  # Model single models
  myBiomodModelOut <- BIOMOD_Modeling(bm.format = myBiomodData,
                                      modeling.id = 'AllModels',
                                      models = c('RF', 'GLM'),
                                      bm.options = myBiomodOptions,
                                      nb.rep = 2,
                                      data.split.perc = 80,
                                      metric.eval = c('TSS','ROC'),
                                      var.import = 3,
                                      do.full.models = FALSE,
                                      seed.val = 42)
}

file.proj <- paste0(myRespName, "/proj_Current/", myRespName, ".Current.projection.out")
if (file.exists(file.proj)) {
  myBiomodProj <- get(load(file.proj))
} else {

  # Project single models
  myBiomodProj <- BIOMOD_Projection(bm.mod = myBiomodModelOut,
                                    proj.name = 'Current',
                                    new.env = myExpl,
                                    models.chosen = 'all',
                                    metric.binary = 'all',
                                    metric.filter = 'all',
                                    build.clamping.mask = TRUE)
}


# ---------------------------------------------------------------
# Load environmental variables extracted from BIOCLIM (bio_3, bio_4, bio_7, bio_11 & bio_12)
myFiles = paste0('external/bioclim/future/bio', c(3, 4, 7, 11, 12), '.grd')
myExplFuture = raster::stack(system.file(myFiles, package = 'biomod2'))
\dontshow{
myExtent <- raster::extent(0,30,45,70)
myExplFuture <- raster::stack(raster::crop(myExplFuture, myExtent))
}
# Project onto future conditions
myBiomodProjectionFuture <- BIOMOD_Projection(bm.mod = myBiomodModelOut,
                                              proj.name = 'Future',
                                              new.env = myExplFuture,
                                              models.chosen = 'all',
                                              metric.binary = 'TSS',
                                              build.clamping.mask = TRUE)

# Load current and future binary projections
proj.current <- raster::stack("GuloGulo/proj_Current/proj_Current_GuloGulo_TSSbin.grd")
proj.future <- raster::stack("GuloGulo/proj_Future/proj_Future_GuloGulo_TSSbin.grd")

# Compute differences
myBiomodRangeSize <- BIOMOD_RangeSize(proj.current = proj.current, proj.future = proj.future)

myBiomodRangeSize$Compt.By.Models
plot(myBiomodRangeSize$Diff.By.Pixel)

# Represent main results 
bm_PlotRangeSize(bm.range = myBiomodRangeSize)


}
\seealso{
\code{\link{BIOMOD_Projection}}, \code{\link{BIOMOD_EnsembleForecasting}}, 
\code{\link{bm_PlotRangeSize}}

Other Main functions: 
\code{\link{BIOMOD_CrossValidation}()},
\code{\link{BIOMOD_EnsembleForecasting}()},
\code{\link{BIOMOD_EnsembleModeling}()},
\code{\link{BIOMOD_FormatingData}()},
\code{\link{BIOMOD_LoadModels}()},
\code{\link{BIOMOD_ModelingOptions}()},
\code{\link{BIOMOD_Modeling}()},
\code{\link{BIOMOD_PresenceOnly}()},
\code{\link{BIOMOD_Projection}()},
\code{\link{BIOMOD_Tuning}()}
}
\author{
Wilfried Thuiller, Damien Georges, Bruno Lafourcade
}
\concept{Main functions}
\keyword{"species}
\keyword{change"}
\keyword{gain}
\keyword{loss}
\keyword{projections}
\keyword{range}
