#' Estimate Semipartial Correlation Coefficients
#' and Generate the Corresponding Sampling Distribution
#' Using Nonparametric Bootstrapping
#'
#' @author Ivan Jacob Agaloos Pesigan
#'
#' @details The vector of semipartial correlation coefficients
#'   (\eqn{r_{s}})
#'   is estimated from bootstrap samples.
#'   Confidence intervals are generated by obtaining
#'   percentiles corresponding to \eqn{100(1 - \alpha)\%}
#'   from the generated sampling
#'   distribution of \eqn{r_{s}},
#'   where \eqn{\alpha} is the significance level.
#'
#' @return Returns an object
#'   of class `betanb` which is a list with the following elements:
#'   \describe{
#'     \item{call}{Function call.}
#'     \item{object}{The function argument `object`.}
#'     \item{thetahatstar}{Sampling distribution of
#'       \eqn{r_{s}}.}
#'     \item{vcov}{Sampling variance-covariance matrix of
#'       \eqn{r_{s}}.}
#'     \item{est}{Vector of estimated
#'       \eqn{r_{s}}.}
#'     \item{fun}{Function used ("SCorMC").}
#'   }
#'
#' @inheritParams BetaNB
#'
#' @examples
#' # Fit the regression model
#' object <- lm(QUALITY ~ NARTIC + PCTGRT + PCTSUPP, data = nas1982)
#' # Generate the sampling distribution of sample covariances
#' # (use a large R, for example, R = 5000 for actual research)
#' nb <- NB(object, R = 50)
#' # Generate confidence intervals for standardized regression slopes
#' rs <- SCorNB(nb)
#' # Methods --------------------------------------------------------
#' print(rs)
#' summary(rs)
#' coef(rs)
#' vcov(rs)
#' confint(rs, level = 0.95)
#' @export
#' @family Beta Nonparametric Bootstrap Functions
#' @keywords betaNB scor
SCorNB <- function(object) {
  stopifnot(
    methods::is(
      object,
      "nb"
    )
  )
  if (object$lm_process$p < 2) {
    stop("Two or more regressors is required.")
  }
  fun <- "SCorNB"
  est <- .SPCor(
    betastar = object$lm_process$betastar,
    sigmacapx = object$lm_process$sigmacapx
  )
  names(est) <- object$lm_process$xnames
  foo <- function(x) {
    return(
      .SPCor(
        betastar = .BetaStarofSigma(
          sigmacap = x,
          q = 1 / sqrt(diag(x)),
          k = object$lm_process$k
        ),
        sigmacapx = x[
          2:object$lm_process$k,
          2:object$lm_process$k,
          drop = FALSE
        ]
      )
    )
  }
  thetahatstar <- lapply(
    X = object$thetahatstar,
    FUN = foo
  )
  vcov <- stats::var(
    do.call(
      what = "rbind",
      args = thetahatstar
    )
  )
  colnames(vcov) <- rownames(vcov) <- names(est)
  out <- list(
    call = match.call(),
    object = object,
    thetahatstar = thetahatstar,
    jackknife = lapply(
      X = object$jackknife,
      FUN = foo
    ),
    vcov = vcov,
    est = est,
    fun = fun
  )
  class(out) <- c(
    "betanb",
    class(out)
  )
  return(
    out
  )
}
