% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mcmc-recover.R
\name{MCMC-recover}
\alias{MCMC-recover}
\alias{mcmc_recover_intervals}
\title{Compare MCMC estimates to "true" parameter values}
\usage{
mcmc_recover_intervals(x, true, batch = rep(1, length(true)),
  facet_args = list(), ..., prob = 0.5, prob_outer = 0.9,
  point_est = c("median", "mean", "none"))
}
\arguments{
\item{x}{A 3-D array, matrix, list of matrices, or data frame of MCMC draws.
The \link{MCMC-overview} page provides details on how to specify each these
allowed inputs.}

\item{true}{A numeric vector of "true" values of the parameters in \code{x}.
There should be one value in \code{true} for each parameter included in
\code{x} and the order of the parameters in \code{true} should be the same
as the order of the parameters in \code{x}.}

\item{batch}{Optionally, a vector-like object (numeric, character, integer,
factor) used to split the parameters into batches. If \code{batch} is
specified, it must have the same length as \code{true} and be in the same
order as \code{true}. Parameters in the same batch will be grouped together
in the same facet in the plot (see the \strong{Examples} section, below).
The default is to group all parameters together into a single batch.
Changing the default is most useful when parameters are on very different
scales, in which case \code{batch} can be used to group them into batches
within which it makes sense to use the same \eqn{y}-axis.}

\item{facet_args}{Arguments (other than \code{facets}) passed to
\code{\link[ggplot2]{facet_wrap}} to control faceting.}

\item{...}{Currently unused.}

\item{prob}{The probability mass to include in the inner interval. The
default is \code{0.5} (50\% interval).}

\item{prob_outer}{The probability mass to include in the outer interval. The
default is \code{0.9} (90\% interval).}

\item{point_est}{The point estimate to show. Either \code{"median"} (the
default), \code{"mean"}, or \code{"none"}.}
}
\value{
A ggplot object that can be further customized using the
  \pkg{ggplot2} package.
}
\description{
Plots comparing MCMC estimates to "true" parameter values. Before fitting a
model to real data it is useful to simulate data according to the model using
known (fixed) parameter values and to check that these "true" parameter
values are (approximately) recovered by fitting the model to the simulated
data. See the \strong{Plot Descriptions} section, below, for details on the
available plots.
}
\section{Plot Descriptions}{

\describe{
  \item{\code{mcmc_recover_intervals}}{
   Central intervals and point estimates computed from MCMC draws, with
   "true" values plotted using a different shape.
  }
}
}
\examples{
\dontrun{
library(rstanarm)
alpha <- 1; beta <- c(-.5, .5); sigma <- 2
X <- matrix(rnorm(200), 100, 2)
y <- rnorm(100, mean = c(alpha + X \%*\% beta), sd = sigma)
fit <- stan_glm(y ~ X)
draws <- as.matrix(fit)
print(colnames(draws))
true <- c(alpha, beta, sigma)
mcmc_recover_intervals(draws, true)

# put the coefficients on X into the same batch
mcmc_recover_intervals(draws, true, batch = c(1, 2, 2, 1))
# equivalent
mcmc_recover_intervals(draws, true, batch = grepl("X", colnames(draws)))
# same but stacked vertically
mcmc_recover_intervals(draws, true,
                       batch = grepl("X", colnames(draws)),
                       facet_args = list(ncol = 1))

# each parameter in its own facet
mcmc_recover_intervals(draws, true, batch = 1:4)
# same but in a different order
mcmc_recover_intervals(draws, true, batch = c(1, 3, 4, 2))
}

}
\seealso{
Other MCMC: \code{\link{MCMC-combos}},
  \code{\link{MCMC-diagnostics}},
  \code{\link{MCMC-distributions}},
  \code{\link{MCMC-intervals}}, \code{\link{MCMC-nuts}},
  \code{\link{MCMC-overview}},
  \code{\link{MCMC-scatterplots}},
  \code{\link{MCMC-traces}}
}

