% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SIPC.AMMI.R
\name{SIPC.AMMI}
\alias{SIPC.AMMI}
\title{Sums of the Absolute Value of the IPC Scores}
\usage{
SIPC.AMMI(model, n, alpha = 0.05, ssi.method = c("farshadfar", "rao"),
  a = 1)
}
\arguments{
\item{model}{The AMMI model (An object of class \code{AMMI} generated by
\code{\link[agricolae]{AMMI}}).}

\item{n}{The number of principal components to be considered for computation.
The default value is the number of significant IPCs.}

\item{alpha}{Type I error probability (Significance level) to be considered
to identify the number of significant IPCs.}

\item{ssi.method}{The method for the computation of simultaneous selection
index. Either \code{"farshadfar"} or \code{"rao"} (See
\code{\link[ammistability]{SSI}}).}

\item{a}{The ratio of the weights given to the stability components for
computation of SSI when \code{method = "rao"} (See
\code{\link[ammistability]{SSI}}).}
}
\value{
A data frame with the following columns:  \item{SIPC}{The SIPC
  values.} \item{SSI}{The computed values of simultaneous selection index for
  yield and stability.} \item{rSIPC}{The ranks of SIPC values.} \item{rY}{The
  ranks of the mean yield of genotypes.} \item{means}{The mean yield of the
  genotypes.}

  The names of the genotypes are indicated as the row names of the data
  frame.
}
\description{
\code{SIPC.AMMI} computes the Sums of the Absolute Value of the IPC Scores
(ASI) (Sneller et al., 1997) considering all significant interaction
principal components (IPCs) in the AMMI model. Using SIPC, the Simultaneous
Selection Index for Yield and Stability (SSI) is also calculated according to
the argument \code{ssi.method}.
}
\details{
The Sums of the Absolute Value of the IPC Scores (\eqn{SIPC}) is computed as
follows:

\ifelse{html}{\out{<p style="text-align: center;"><em>SIPC =
&sum;<sup>N'</sup><sub style="line-height: 1.8; margin-left:
-3ex;">n=1</sub>|&lambda;<sup>0.5</sup><sub style="line-height: 1.8;
margin-left: -2ex;">n</sub>
&gamma;<sub>in</sub>|</em></p></body>}}{\deqn{SIPC = \sum_{n=1}^{N'} \left |
\lambda_{n}^{0.5}\gamma_{in} \right |}}

OR

\ifelse{html}{\out{<p style="text-align: center;"><em>SIPC =
&sum;<sup>N'</sup><sub style="line-height: 1.8; margin-left:
-2ex;">n=1</sub>|PC<sub>n</sub>|</em></p>}}{\deqn{SIPC = \sum_{n=1}^{N'}\left
| PC_{n} \right |}}

Where, \ifelse{html}{\out{<i>N'</i>}}{\eqn{N'}} is the number of significant
IPCs (number of IPC that were retained in the AMMI model via F tests);
\ifelse{html}{\out{<i>&lambda;<sub>n</sub></i>}}{\eqn{\lambda_{n}}} is the
singular value for \ifelse{html}{\out{<i>n</i>}}{\eqn{n}}th IPC and
correspondingly \ifelse{html}{\out{<i>&lambda;<sup>2</sup><sub
style="line-height: 1.8; margin-left:
-1ex;">n</sub></i>}}{\eqn{\lambda_{n}^{2}}} is its eigen value;
\ifelse{html}{\out{<i>&gamma;<sub>in<sub></i>}}{\eqn{\gamma_{in}}} is the
eigenvector value for \ifelse{html}{\out{<i>i</i>}}{\eqn{i}}th genotype; and
\ifelse{html}{\out{<i>PC<sub>1</sub></i>}}{\eqn{PC_{1}}},
\ifelse{html}{\out{<i>PC<sub>2</sub></i>}}{\eqn{PC_{2}}},
\ifelse{html}{\out{&hellip;}}{\eqn{\cdots}},
\ifelse{html}{\out{<i>PC<sub>n</sub></i>}}{\eqn{PC_{n}}} are the scores of
1st, 2nd, ..., and \ifelse{html}{\out{<i>n</i>}}{\eqn{n}}th IPC.

The closer the SIPC scores are to zero, the more stable the genotypes are
across test environments.
}
\examples{
library(agricolae)
data(plrv)

# AMMI model
model <- with(plrv, AMMI(Locality, Genotype, Rep, Yield, console = FALSE))

# ANOVA
model$ANOVA

# IPC F test
model$analysis

# Mean yield and IPC scores
model$biplot

# G*E matrix (deviations from mean)
array(model$genXenv, dim(model$genXenv), dimnames(model$genXenv))

# With default n (N') and default ssi.method (farshadfar)
SIPC.AMMI(model)

# With n = 4 and default ssi.method (farshadfar)
SIPC.AMMI(model, n = 4)

# With default n (N') and ssi.method = "rao"
SIPC.AMMI(model, ssi.method = "rao")

# Changing the ratio of weights for Rao's SSI
SIPC.AMMI(model, ssi.method = "rao", a = 0.43)

}
\references{
\insertRef{sneller_repeatability_1997}{ammistability}
}
\seealso{
\code{\link[agricolae]{AMMI}}, \code{\link[ammistability]{SSI}}
}
