% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scores.grid.time.R
\name{scores.grid.time}
\alias{scores.grid.time}
\title{Scores for gridded reference data with a varying time dimension}
\usage{
scores.grid.time(long.name, nc.mod, nc.ref, mod.id, ref.id, unit.conv.mod,
  unit.conv.ref, variable.unit, score.weights = c(1, 2, 1, 1, 1),
  outlier.factor = 1000, irregular = FALSE,
  my.projection = "+proj=ob_tran +o_proj=longlat +o_lon_p=83. +o_lat_p=42.5 +lon_0=263.",
  numCores = 2, timeInt = "month", outputDir = FALSE)
}
\arguments{
\item{long.name}{A string that gives the full name of the variable, e.g. 'Gross primary productivity'}

\item{nc.mod}{A string that gives the path and name of the netcdf file that contains the model output, e.g. '/home/model_gpp.nc'}

\item{nc.ref}{A string that gives the path and name of the netcdf file that contains the reference data output, e.g. '/home/reference_gpp.nc'}

\item{mod.id}{A string that identifies the source of the reference data set, e.g. 'CLASSIC'}

\item{ref.id}{A string that identifies the source of the reference data set, e.g. 'MODIS'}

\item{unit.conv.mod}{A number that is used as a factor to convert the unit of the model data, e.g. 86400}

\item{unit.conv.ref}{A number that is used as a factor to convert the unit of the reference data, e.g. 86400}

\item{variable.unit}{A string that gives the final units using LaTeX notation, e.g. 'gC m$^{-2}$ day$^{-1}$'}

\item{score.weights}{R object that gives the weights of each score (\eqn{S_{bias}}, \eqn{S_{rmse}}, \eqn{S_{phase}}, \eqn{S_{iav}}, \eqn{S_{dist}})
that are used for computing the overall score, e.g. c(1,2,1,1,1)}

\item{outlier.factor}{A number that is used to define outliers, e.g. 10.
Plotting raster objects that contain extreme outliers lead to figures where
most grid cells are presented by a single color since the color legend covers
the entire range of values. To avoid this, the user may define outliers that
will be masked out and marked with a red dot. Outliers are all values that
exceed the interquartile range multiplied by the outlier factor defined here.}

\item{irregular}{Logical. If TRUE the data is converted from an irregular to a regular grid. Default is FALSE.}

\item{my.projection}{A string that defines the projection of the irregular grid}

\item{numCores}{An integer that defines the number of cores, e.g. 2}

\item{timeInt}{A string that gives the time interval of the model data, e.g. 'month' or 'year'}

\item{outputDir}{A string that gives the output directory, e.g. '/home/project/study'. The output will only be written if the user specifies an output directory.}
}
\value{
(1) A list that contains three elements. The first element is a a
raster stack with model data
(mean, \eqn{mod.mean}; interannual-variability, \eqn{mod.iav}; month of annual cycle maximum, \eqn{mod.max.month}),
the reference data (mean, \eqn{ref.mean}; interannual-variability, \eqn{ref.iav}; month of annual cycle maximum, \eqn{ref.max.month}),
statistical metrics
(bias, \eqn{bias}; centralized root mean square error, \eqn{crmse}; time difference of the annual cycle maximum, \eqn{phase}),
and scores (bias score, \eqn{bias.score}; root mean square error score, \eqn{rmse.score}; inter-annual variability score \eqn{iav.score}; annual cycle score (\eqn{phase.score}).
The second and third element of the list are spatial
point data frames that give the model and reference outliers, respectively.
The content of the list can be plotted using \link{plotGrid}.

(2) NetCDF files for each of the statistical variables listed above.

(3) Three text files: (i) score values and (ii) score inputs averaged across
the entire grid, and (iii) score values for each individual grid cell.
}
\description{
This function compares model output against remote-sensing
based reference data that vary in time. The performance of a model is
expressed through scores that range from zero to one, where increasing values
imply better performance. These scores are computed in five steps:
\eqn{(i)} computation of a statistical metric,
\eqn{(ii)} nondimensionalization,
\eqn{(iii)} conversion to unit interval,
\eqn{(iv)} spatial integration, and
\eqn{(v)} averaging scores computed from different statistical metrics.
The latter includes the bias, root-mean-square error, phase shift,
inter-annual variability, and spatial distribution. The corresponding equations
are documented in \code{\link{amber-package}}.
}
\examples{
\donttest{
library(amber)
library(doParallel)
library(foreach)
library(ncdf4)
library(parallel)
library(raster)

# (1) Global plots on a regular grid
long.name <- 'Gross primary productivity'
nc.mod <- system.file('extdata/modelRegular', 'gpp_monthly.nc', package = 'amber')
nc.ref <- system.file('extdata/referenceRegular', 'gpp_GBAF_128x64.nc', package = 'amber')
mod.id <- 'CLASSIC' # define a model experiment ID
ref.id <- 'GBAF' # give reference dataset a name
unit.conv.mod <- 86400*1000 # optional unit conversion for model data
unit.conv.ref <- 86400*1000 # optional unit conversion for reference data
variable.unit <- 'gC m$^{-2}$ day$^{-1}$' # unit after conversion (LaTeX notation)

# Short version using default settings:
plot.me <- scores.grid.time(long.name, nc.mod, nc.ref, mod.id, ref.id, unit.conv.mod,
unit.conv.ref, variable.unit)
plotGrid(long.name, plot.me)

# Additional parameters:
score.weights <- c(1,2,1,1,1) # score weights of S_bias, S_rmse, S_phase, S_iav, S_dist
outlier.factor <- 1
irregular <- FALSE
my.projection <- '+proj=ob_tran +o_proj=longlat +o_lon_p=83. +o_lat_p=42.5 +lon_0=263.'
numCores <- 2

plot.me <- scores.grid.time(long.name, nc.mod, nc.ref, mod.id, ref.id, unit.conv.mod,
unit.conv.ref, variable.unit, score.weights, outlier.factor, irregular,
my.projection, numCores)
plotGrid(long.name, plot.me)

# (2) Regional plots on a rotated grid
long.name <- 'Gross primary productivity'
nc.mod <- system.file('extdata/modelRotated', 'gpp_monthly.nc', package = 'amber')
nc.ref <- system.file('extdata/referenceRotated', 'gpp_GBAF_rotated.nc', package = 'amber')
mod.id <- 'CLASSIC' # define a model experiment ID
ref.id <- 'GBAF' # give reference dataset a name
unit.conv.mod <- 86400*1000 # optional unit conversion for model data
unit.conv.ref <- 86400*1000 # optional unit conversion for reference data
variable.unit <- 'gC m$^{-2}$ day$^{-1}$' # unit after conversion (LaTeX notation)
score.weights <- c(1,2,1,1,1) # score weights of S_bias, S_rmse, S_phase, S_iav, S_dist
outlier.factor <- 10
irregular <- TRUE
my.projection <- '+proj=ob_tran +o_proj=longlat +o_lon_p=83. +o_lat_p=42.5 +lon_0=263.'
numCores <- 2

plot.me <- scores.grid.time(long.name, nc.mod, nc.ref, mod.id, ref.id, unit.conv.mod,
unit.conv.ref, variable.unit, score.weights, outlier.factor, irregular,
my.projection, numCores)

# Plot results:
irregular <- TRUE # data is on an irregular grid
my.projection <- '+proj=ob_tran +o_proj=longlat +o_lon_p=83. +o_lat_p=42.5 +lon_0=263.'
shp.filename <- system.file('extdata/ne_50m_admin_0_countries/ne_50m_admin_0_countries.shp',
 package = 'amber')
my.xlim <- c(-171, 0) # longitude range that you wish to plot
my.ylim <- c(32, 78) # latitude range that you wish to plot
plot.width <- 7 # plot width
plot.height <- 3.8 # plot height

plotGrid(long.name, plot.me, irregular, my.projection,
shp.filename, my.xlim, my.ylim, plot.width, plot.height)
}
}
