% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aba_selection.R
\name{aba_selection}
\alias{aba_selection}
\title{Run model selection on an aba model.}
\usage{
aba_selection(
  model,
  method = c("forward", "backward"),
  criteria = c("aic", "pval"),
  threshold = NULL,
  verbose = FALSE
)
}
\arguments{
\item{model}{abaModel. The fitted aba model to run selection on.}

\item{method}{string. The selection algorithm to use (forward or backward).}

\item{criteria}{string. Which metric to use when selecting the next
model (aic or pval).}

\item{threshold}{numeric. Which threshold to use for the selected metric
(defaults to -2 for aic; defaults to 0.1 for pval).}

\item{verbose}{logical. Whether to print out results of each selection round.}
}
\value{
an abaSelection object which contains model summary information such
as coefficients and metrics for each selection round across the different
groups/outcomes/stats.
}
\description{
This function allows you to run model selection on a fitted aba model. The
function supports both forward and backward selection algorithms, both AIC
and p-value as selection criteria, and arbitrary thresholds.
}
\details{
Forward selection starts from covariates-only and tests the addition of all
predictor sets individually, then adds the predictor set which improves the
model criteria the most. Backward selection starts from the inclusion of
all covariates + predictor sets and tests the removal of all predictor sets
individually, then removes the predictor set which improves the model criteria
the most. If there are no predictor sets whose addition/removal results in
an improvement in the selected criteria by a value at least as good as the
selected threshold, then the selection stops and the current model is frozen.
Also, note that the model selection procedure is run separately for each
group - outcome - stat combination.
}
\examples{

df <- aba::adnimerge \%>\% dplyr::filter(VISCODE == 'bl')

# standard model selection
model <- df \%>\% aba_model() \%>\%
  set_outcomes(ConvertedToAlzheimers) \%>\%
  set_predictors(
    CDRSB_bl, ADAS13_bl, MMSE_bl,
    CSF_ABETA_bl, CSF_PTAU_bl, CSF_TAU_bl,
    PLASMA_ABETA_bl, PLASMA_PTAU181_bl, PLASMA_NFL_bl,
    MRI_HIPP_bl,
    PET_ABETA_bl
  ) \%>\%
  set_covariates(AGE, GENDER, EDUCATION) \%>\%
  set_stats('glm') \%>\%
  aba_fit()

model_summary <- model \%>\% aba_summary()

# default selection - forward selection by AIC with threshold = -2
\donttest{
model_selection <- model \%>\% aba_selection(verbose=TRUE)
}
# selection with p-value and threshold = 0.1
\donttest{
model_selection <- model \%>\%
  aba_selection(criteria = 'pval', threshold=0.1, verbose=TRUE)
}
# selection by group
model2 <- model \%>\%
  set_predictors(
    c(CDRSB_bl,ADAS13_bl,MMSE_bl),
    c(CSF_ABETA_bl,CSF_PTAU_bl,CSF_TAU_bl),
    c(PLASMA_ABETA_bl, PLASMA_PTAU181_bl, PLASMA_NFL_bl),
    c(MRI_HIPP_bl),
    c(PET_ABETA_bl)
  ) \%>\%
  aba_fit()

model_summary2 <- model2 \%>\% aba_summary()

\donttest{
model_selection2 <- model2 \%>\%
  aba_selection(criteria='pval', threshold=0.1, verbose=TRUE)
}

# add more outcomes
model3 <- model2 \%>\%
  set_outcomes(ConvertedToAlzheimers, ConvertedToDementia) \%>\%
  aba_fit()

\donttest{
model_selection3 <- model3 \%>\%
  aba_selection(criteria='pval', threshold=0.1, verbose=TRUE)
}

# add more groups
model4 <- model3 \%>\%
  set_groups(everyone(), DX_bl \%in\% c('MCI','AD')) \%>\%
  aba_fit()

\donttest{
model_selection4 <- model4 \%>\%
  aba_selection(criteria='pval', threshold=0.1, verbose=TRUE)
}

}
