# arxiv_open
#
#' Open abstract for results of arXiv search
#'
#' Open, in web browser, the abstract pages for each of set of arXiv search results.
#'
#' @param search_results Data frame of search results, as returned from [arxiv_search()].
#' @param limit Maximum number of abstracts to open in one call.
#'
#' @details There is a delay between calls to
#' [utils::browseURL()], with the amount taken from the R
#' option `"aRxiv_delay"` (in seconds); if missing, the default
#' is 3 sec.
#'
#' @return (Invisibly) Vector of character strings with URLs of
#' abstracts opened.
#'
#' @seealso [arxiv_search()]
#'
#' @examples
#' \donttest{z <- arxiv_search('au:"Peter Hall" AND ti:deconvolution')
#' arxiv_open(z)}
#'
#' @export
arxiv_open <-
function(search_results, limit=20)
{
    stopifnot(limit >= 1)

    if(!is.data.frame(search_results))
        stop("Input should be a data frame, as returned from arxiv_search()")

    if(nrow(search_results) == 0) {
        warning("Input has 0 rows")
        return(invisible(NULL))
    }

    links <- search_results$link_abstract
    links <- links[links != ""]
    if(length(links) > limit) {
        warning("More abstracts (", length(links), ") than maximum to be opened (", limit, ").")
        links <- links[1:limit]
    }

    for(link in links) {
        delay_if_necessary()
        utils::browseURL(link)
    }

    invisible(links)
}
