% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/run_sim_parallel.R
\name{run_sim_parallel}
\alias{run_sim_parallel}
\title{Run simulations in parallel mode (at the simulation level)}
\usage{
run_sim_parallel(
  arm_list = c("int", "noint"),
  sensitivity_inputs = NULL,
  common_all_inputs = NULL,
  common_pt_inputs = NULL,
  unique_pt_inputs = NULL,
  init_event_list = NULL,
  evt_react_list = evt_react_list,
  util_ongoing_list = NULL,
  util_instant_list = NULL,
  util_cycle_list = NULL,
  cost_ongoing_list = NULL,
  cost_instant_list = NULL,
  cost_cycle_list = NULL,
  other_ongoing_list = NULL,
  other_instant_list = NULL,
  npats = 500,
  n_sim = 1,
  psa_bool = NULL,
  sensitivity_bool = FALSE,
  sensitivity_names = NULL,
  n_sensitivity = 1,
  ncores = 1,
  input_out = NULL,
  ipd = 1,
  timed_freq = NULL,
  debug = FALSE,
  accum_backwards = FALSE,
  continue_on_error = FALSE,
  seed = NULL
)
}
\arguments{
\item{arm_list}{A vector of the names of the interventions evaluated in the simulation}

\item{sensitivity_inputs}{A list of sensitivity inputs that do not change within a sensitivity in a similar fashion to common_all_inputs, etc}

\item{common_all_inputs}{A list of inputs common across patients that do not change within a simulation}

\item{common_pt_inputs}{A list of inputs that change across patients but are not affected by the intervention}

\item{unique_pt_inputs}{A list of inputs that change across each intervention}

\item{init_event_list}{A list of initial events and event times. If no initial events are given, a "Start" event at time 0 is created automatically}

\item{evt_react_list}{A list of event reactions}

\item{util_ongoing_list}{Vector of QALY named variables that are accrued at an ongoing basis (discounted using drq)}

\item{util_instant_list}{Vector of QALY named variables that are accrued instantaneously at an event (discounted using drq)}

\item{util_cycle_list}{Vector of QALY named variables that are accrued in cycles (discounted using drq)}

\item{cost_ongoing_list}{Vector of cost named variables that are accrued at an ongoing basis (discounted using drc)}

\item{cost_instant_list}{Vector of cost named variables that are accrued instantaneously at an event (discounted using drc)}

\item{cost_cycle_list}{Vector of cost named variables that are accrued in cycles (discounted using drc)}

\item{other_ongoing_list}{Vector of other named variables that are accrued at an ongoing basis (discounted using drq)}

\item{other_instant_list}{Vector of other named variables that are accrued instantaneously at an event (discounted using drq)}

\item{npats}{The number of patients to be simulated (it will simulate npats * length(arm_list))}

\item{n_sim}{The number of simulations to run per sensitivity}

\item{psa_bool}{A boolean to determine if PSA should be conducted. If n_sim > 1 and psa_bool = FALSE, the differences between simulations will be due to sampling}

\item{sensitivity_bool}{A boolean to determine if Scenarios/DSA should be conducted.}

\item{sensitivity_names}{A vector of scenario/DSA names that can be used to select the right sensitivity (e.g., c("Scenario_1", "Scenario_2")). The parameter "sens_name_used" is created from it which corresponds to the one being used for each iteration.}

\item{n_sensitivity}{Number of sensitivity analysis (DSA or Scenarios) to run. It will be interacted with sensitivity_names argument if not null (n_sensitivityitivity = n_sensitivity * length(sensitivity_names)). For DSA, it should be as many parameters as there are. For scenario, it should be 1.}

\item{ncores}{The number of cores to use for parallel computing}

\item{input_out}{A vector of variables to be returned in the output data frame}

\item{ipd}{Integer taking value 0 if no IPD data returned, 1 for full IPD data returned, and 2 IPD data but aggregating events}

\item{timed_freq}{If NULL, it does not produce any timed outputs. Otherwise should be a number (e.g., every 1 year)}

\item{debug}{If TRUE, will generate a log file}

\item{accum_backwards}{If TRUE, the ongoing accumulators will count backwards (i.e., the current value is applied until the previous update). If FALSE, the current value is applied between the current event and the next time it is updated. If TRUE, user must use \code{modify_item} and \code{modify_item_seq} or results will be incorrect.}

\item{continue_on_error}{If TRUE, on error  at patient stage will attempt to continue to the next simulation (only works if n_sim and/or n_sensitivity are > 1, not at the patient level)}

\item{seed}{Starting seed to be used for the whole analysis. If null, it's set to 1 by default.}
}
\value{
A list of lists with the analysis results
}
\description{
Run simulations in parallel mode (at the simulation level)
}
\details{
This function is slightly different from \code{run_sim}.
\code{run_sim} allows to run single-core.
\code{run_sim_parallel} allows to use multiple-core at the simulation level,
making it more efficient for a large number of simulations relative to \code{run_sim} (e.g., for  PSA).

Event ties are processed in the order declared within the \code{init_event_list} argument (\code{evts} argument within the first sublist of that object).
To do so, the program automatically adds a sequence from to 0 to the (number of events - 1) times 1e-10 to add to the event times when selecting the event with minimum time.
This time has been selected as it's relatively small yet not so small as to be ignored by which.min (see .Machine for more details)

A list of protected objects that should not be used by the user as input names or in the global environment to avoid the risk of overwriting them is as follows:
c("arm", "arm_list", "categories_for_export", "cur_evtlist", "curtime", "evt", "i", "prevtime", "sens", "simulation", "sens_name_used","list_env","uc_lists","npats","ipd").

The engine uses the L'Ecuyer-CMRG for the random number generator.
Note that if ncores > 1, then results per simulation will only be exactly replicable if using run_sim_parallel
(as seeds are automatically transformed to be seven integer seeds -i.e, L'Ecuyer-CMRG seeds-)
Note that the random seeds are set to be unique in their category (i.e., at patient level, patient-arm level, etc.)

If no \code{drc} or \code{drq} parameters are passed within \code{sensitivity} or \code{common_all} input lists, these are assigned a default value 0.03 for discounting costs, QALYs and others.

Ongoing items will look backward to the last time updated when performing the discounting and accumulation.
This means that the user does not necessarily need to keep updating the value, but only add it when the value
changes looking forward (e.g., o_q = utility at event 1, at event 2 utility does not change, but at event 3 it does,
so we want to make sure to add o_q = utility at event 3 before updating utility. The program will automatically
look back until event 1). Note that in previous versions of the package backward was the default, and now this has switched to forward.

If using \code{accum_backwards = TRUE}, then it is mandatory for the user to use \code{modify_item} and \code{modify_item_seq} in event reactions,
as the standard assignment approach (e.g., \code{a <- 5}) will not calculate the right results, particularly in the presence of
conditional statements.

If the \code{cycle} lists are used, then it is expected the user will declare as well the name of the variable
pasted with \code{cycle_l} and \code{cycle_starttime} (e.g., c_default_cycle_l and c_default_cycle_starttime) to
ensure the discounting can be computed using cycles, with cycle_l being the cycle length, and cycle_starttime
being the starting time in which the variable started counting.

\code{debug = TRUE} will export a log file with the timestamp up the error in the main working directory. Note that
using this mode without modify_item or modify_item_seq may lead to inaccuracies if assignments are done in non-standard ways,
as the AST may not catch all the relevant assignments (e.g., an assigment like assign(paste("x_",i),5)
in a loop will not be identified, unless using modify_item()).

If \code{continue_on_error} is set to FALSE, it will only export analysis level inputs due to the parallel engine
(use single-engine for those inputs) \code{continue_on_error} will skip the current simulation
(so it won't continue for the rest of patient-arms) if TRUE.
Note that this will make the progress bar not correct, as a set of patients that were expected to be run is not.
}
\examples{
library(magrittr)
common_all_inputs <-add_item(
util.sick = 0.8,
util.sicker = 0.5,
cost.sick = 3000,
cost.sicker = 7000,
cost.int = 1000,
coef_noint = log(0.2),
HR_int = 0.8,
drc = 0.035, #different values than what's assumed by default
drq = 0.035,
random_seed_sicker_i = sample.int(100000,5,replace = FALSE)
)

common_pt_inputs <- add_item(death= max(0.0000001,rnorm(n=1, mean=12, sd=3))) 

unique_pt_inputs <- add_item(fl.sick = 1,
                             q_default = util.sick,
                             c_default = cost.sick + if(arm=="int"){cost.int}else{0}) 
                             
init_event_list <- 
add_tte(arm=c("noint","int"), evts = c("sick","sicker","death") ,input={
  sick <- 0
  sicker <- draw_tte(1,dist="exp",
   coef1=coef_noint, beta_tx = ifelse(arm=="int",HR_int,1),
   seed = random_seed_sicker_i[i])
  
})   

evt_react_list <-
add_reactevt(name_evt = "sick",
             input = {}) \%>\%
  add_reactevt(name_evt = "sicker",
               input = {
                 modify_item(list(q_default = util.sicker,
                                  c_default = cost.sicker + if(arm=="int"){cost.int}else{0},
                                  fl.sick = 0)) 
               }) \%>\%
  add_reactevt(name_evt = "death",
               input = {
                 modify_item(list(q_default = 0,
                                  c_default = 0, 
                                  curtime = Inf)) 
               }) 
               
util_ongoing <- "q_default"
cost_ongoing <- "c_default"
                          

run_sim_parallel(arm_list=c("int","noint"),
common_all_inputs = common_all_inputs,
common_pt_inputs = common_pt_inputs,
unique_pt_inputs = unique_pt_inputs,
init_event_list = init_event_list,
evt_react_list = evt_react_list,
util_ongoing_list = util_ongoing,
cost_ongoing_list = cost_ongoing,
npats = 2,
n_sim = 1,
psa_bool = FALSE,
ipd = 1,
ncores = 1)

}
