\name{disscenter}
\alias{disscenter}
\title{Compute distances to the center of a group}

\description{
Computes the dissimilarity between objects and their group center from their pairwise dissimilarity matrix.
}

\usage{
disscenter(diss, group=NULL, medoids.index=NULL,
           allcenter = FALSE, weights=NULL, squared=FALSE)
}

\arguments{
   \item{diss}{a dissimilarity matrix such as generated by \code{\link{seqdist}}, or a
	\code{dist} object (see \code{\link{dist}})}

   \item{group}{if \code{NULL} (default), the whole data set is considered. Otherwise a different center is considered for each distinct value of the group variable}

% \item{trim}{percentage of individuals that are considered as
%outliers and excluded from the computation of distance to center}

   \item{medoids.index}{if \code{NULL}, returns the dissimilarity to the center.
If set to \code{"first"}, returns the index of the first encountered
most central sequence. If \code{group} is set, an index is returned per group. When set to \code{"all"}, indexes of all medoids (one list per group) are returned.}

   \item{allcenter}{logical. If \code{TRUE}, returns a \code{data.frame}
containing the dissimilarity between each object and its group
center, each column corresponding to a group.}

    \item{weights}{optional numerical vector containing weights.}
	\item{squared}{Logical. If \code{TRUE} \code{diss} is squared.}

% \item{max.iter}{maximum number of iteration to trim the outliers}
 }

\details{
	This function computes the dissimilarity between given objects and their group center. It is possible that the group center does not belong to the space formed by the objects (in the same way as the average of integer numbers is not necessarily an integer itself).
	This distance can also be understood as the contribution to the discrepancy (see \code{\link{dissvar}}).
	Note that when the dissimilarity measure does not respect the triangle inequality, the dissimilarity between a given object and its group center may be negative

	It can be shown that this dissimilarity is equal to (see \cite{Batagelj 1988}):
		\deqn{d_{x\tilde{g}}=\frac{1}{n}\big(\sum_{i=1}^{n}d_{xi}-SS\big)}{d_(xg)=1/n *(sum d_xi - SS)}
	where \eqn{SS} is the sum of squares (see \code{\link{dissvar}}).
}
\value{
	A vector with the dissimilarity to the group center for each object, or a list of medoid indexes.
}

\references{
    Studer, M., G. Ritschard, A. Gabadinho and N. S. Müller (2011). Discrepancy analysis of state sequences, \emph{Sociological Methods and Research}, Vol. 40(3), 471-510, \doi{10.1177/0049124111415372}.

  Studer, M., G. Ritschard, A. Gabadinho and N. S. Müller (2010)
  Discrepancy	analysis of complex objects using dissimilarities.
  In F. Guillet, G. Ritschard, D. A. Zighed and H. Briand (Eds.),
  \emph{Advances in Knowledge Discovery and Management},
  Studies in Computational Intelligence, Volume 292, pp. 3-19. Berlin: Springer.

	Studer, M., G. Ritschard, A. Gabadinho and N. S. Müller (2009)
	Analyse de dissimilarités par arbre d'induction. In EGC 2009,
	\emph{Revue des Nouvelles Technologies de l'Information}, Vol. E-15, pp. 7--18.

	Batagelj, V. (1988) Generalized ward and related clustering problems. In H. Bock (Ed.),
	\emph{Classification and related methods of data analysis}, Amsterdam: North-Holland, pp. 67--74.

}

\seealso{
  \code{\link{dissvar}} to compute the pseudo variance from dissimilarities and for a basic introduction to concepts of pseudo variance analysis
  \cr
  \code{\link{dissassoc}} to test association between objects represented by their dissimilarities and a covariate.
  \cr
  \code{\link{disstree}} for an induction tree analyse of objects characterized by a dissimilarity matrix.
  \cr
  \code{\link{dissmfacw}} to perform multi-factor analysis of variance from pairwise dissimilarities.
}

\examples{
## Defining a state sequence object
data(mvad)
mvad.seq <- seqdef(mvad[, 17:86])

## Building dissimilarities (any dissimilarity measure can be used)
mvad.ham <- seqdist(mvad.seq, method="HAM")

## Compute distance to center according to group gcse5eq
dc <- disscenter(mvad.ham, group=mvad$gcse5eq)

## Ploting distribution of dissimilarity  to center
boxplot(dc~mvad$gcse5eq, col="cyan")

## Retrieving index of the first medoids, one per group
dc <- disscenter(mvad.ham, group=mvad$Grammar, medoids.index="first")
print(dc)

## Retrieving index of all medoids in each group
dc <- disscenter(mvad.ham, group=mvad$Grammar, medoids.index="all")
print(dc)
}
\author{Matthias Studer (with Gilbert Ritschard for the help page)}
% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
\keyword{Dissimilarity-based analysis}
