###########################################
#THIS FILE CONTAINS
#1) function for building unordered couples
#2) function for building ordered couples
###########################################


###########################################
#1) unordered

#'Build pairs (unordered couples) in a dataset.
#'
#'This function builds and counts the number of all types of pairs in a data vector or matrix,
#'according to a chosen adjacency matrix.
#'
#'This function needs a data matrix or data vector of any type (numeric, factor, character, ...),
#'and an adjacency matrix as generated by [adj_mat()]. It returns all the data pairs
#'identified by the adjacency matrix, i.e. occurring at the chosen neighbourhood distance.
#'Relative and absolute frequencies for all possible pairs are returned, and may be used
#'for computation of spatial entropy at the chosen distance range.
#''Unordered couple', i.e. 'pair', means that the relative spatial location is irrelevant, i.e. that a couple
#'where category \eqn{i} occurs at the left of category \eqn{j} is identical to a couple
#'where category \eqn{j} occurs at the left of category \eqn{i}.
#'
#' @param data A matrix or vector, can be numeric, factor, character...
#'   If the dataset is a point pattern, `data` is the mark vector.
#' @param adj.mat An adjacency matrix, upper- or lower-triangular. Provided by user or generated by [adj_mat()].
#' @param missing.cat Optional, a vector with the names of all categories that are absent in `data`.
#'
#' @return The number of pairs, and a table with absolute and relative frequencies
#'  for each pair of categories.
#'
#' @examples
#' dist.mat=euclid_dist(cbind(rep(1:5, each=5), rep(1:5,5)))
#' adj.mat=adj_mat(dist.mat, dd1=dist.mat[1,2]) #for the contiguity matrix
#' pair_count(sample(1:3, 25, replace=TRUE), adj.mat)
#'
#' @export

pair_count=function(data, adj.mat, missing.cat=NULL)
{
  couplevec=NULL
  datavec=c(data)
  for(j in 1:length(datavec))
  {
    ind=which(adj.mat[j,]==1)
    if (length(ind)>0) {
      assoc=datavec[ind]
      assoc.num=as.numeric(as.factor(datavec))[ind]
      couple=ifelse(as.numeric(as.factor(datavec))[j]<=assoc.num,
                    paste(datavec[j], assoc, sep=""),
                    paste(assoc, datavec[j], sep=""))} else couple=NULL
                    couplevec=c(couplevec, couple)
  }

  if (is.numeric(c(data)))
    cat.names=sort(c(unique(c(data)), missing.cat))
  if (is.character(c(data))|is.factor(c(data)))
    cat.names=sort(as.factor(c(unique(c(data)), as.character(missing.cat))))
  couple.names=NULL
  for(i in 1:length(cat.names))
    couple.names=c(couple.names, paste(cat.names[i], cat.names[i:length(cat.names)], sep=""))

  couple.n=length(couplevec)
  #build relative frequencies
  couple.list=sort(unique(couplevec))
  abs.freq=as.numeric(table(couplevec))
  abs.freq.complete=numeric(length(couple.names))
  for(cc in 1:length(couple.names))
  {
    which.ind=which(couple.list==couple.names[cc])
    if (length(which.ind)>0) abs.freq.complete[cc]=abs.freq[which.ind]
  }
  rel.freq=abs.freq.complete/couple.n
  freq.table=data.frame(couple.names, abs.freq.complete, rel.freq)
  colnames(freq.table)=c("couple", "abs.frequency", "proportion")
  return(list(probabilities=freq.table, Qk=couple.n))
}
###########################################


###########################################
#2) ordered

#'Build ordered couples in a dataset.
#'
#'This function builds and counts the number of all types of ordered couples in a data vector or matrix,
#'according to a chosen adjacency matrix. 'Ordered' means that a couple (i,j) is
#'different than a couple (j,i).
#'
#'This function needs a data matrix or data vector of any type (numeric, factor, character, ...),
#'and an adjacency matrix as generated by [adj_mat()]. It returns all the data couples
#'identified by the adjacency matrix, i.e. occurring at the chosen neighbourhood distance.
#'Relative and absolute frequencies for all possible couples are returned, and may be used
#'for computation of spatial entropy at the chosen distance range.
#''Ordered' means that the relative spatial location is relevant, i.e. that a couple
#'where category \eqn{i} occurs at the left of category \eqn{j} is different from a couple
#'where category \eqn{j} occurs at the left of category \eqn{i}.
#'
#' @param data A matrix or vector, can be numeric, factor, character...
#'   If the dataset is a point pattern, `data` is the mark vector.
#' @param adj.mat An adjacency matrix, upper- or lower-triangular. Provided by user or generated by \code{\link{adj_mat}}.
#' @param missing.cat A vector with the names of all categories that are absent in `data`.
#'
#' @return The number of couples, and a table with absolute and relative frequencies
#'  for each couple of categories.
#'
#' @examples
#' dist.mat=euclid_dist(cbind(rep(1:5, each=5), rep(1:5,5)))
#' adj.mat=adj_mat(dist.mat, dd1=dist.mat[1,2]) #for the contiguity matrix
#' couple_count(sample(1:3, 25, replace=TRUE), adj.mat)
#'
#' @export


couple_count=function(data, adj.mat, missing.cat=NULL)
{
  couplevec=NULL
  datavec=c(data)
  for(j in 1:length(datavec))
  {
    ind=which(adj.mat[j,]==1)
    if (length(ind)>0) {assoc=datavec[ind]
    couple=paste(datavec[j], assoc, sep="")} else couple=NULL
    couplevec=c(couplevec, couple)
  }

  if (is.numeric(c(data)))
    cat.names=sort(c(unique(c(data)), missing.cat))
  if (is.character(c(data))|is.factor(c(data)))
    cat.names=sort(as.factor(c(unique(c(data)), as.character(missing.cat))))
  couple.names=NULL
  for(i in 1:length(cat.names))
    couple.names=c(couple.names, paste(cat.names[i], cat.names[1:length(cat.names)], sep=""))

  #paste

  couple.n=length(couplevec)
  #build relative frequencies
  couple.list=sort(unique(couplevec))
  abs.freq=as.numeric(table(couplevec))
  abs.freq.complete=numeric(length(couple.names))
  for(cc in 1:length(couple.names))
  {
    which.ind=which(couple.list==couple.names[cc])
    if (length(which.ind)>0) abs.freq.complete[cc]=abs.freq[which.ind]
  }
  rel.freq=abs.freq.complete/couple.n
  freq.table=data.frame(couple.names, abs.freq.complete, rel.freq)
  colnames(freq.table)=c("couple", "abs.frequency", "proportion")
  return(list(probabilities=freq.table, Qk=couple.n))
}
###########################################

