% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PedSim_Functions.R
\name{sim_ped}
\alias{sim_ped}
\title{Simulate a pedigree}
\usage{
sim_ped(hazard_rates, GRR, FamID, founder_byears, stop_year = NULL,
  carrier_prob = 0.002, RVfounder = FALSE, birth_range = c(18, 45),
  NB_params = c(2, 4/7), fert = 1)
}
\arguments{
\item{hazard_rates}{An object of class \code{hazard}, created by \code{\link{hazard}}.}

\item{GRR}{Numeric. The genetic relative-risk of disease, i.e. the relative-risk of disease for individuals who carry at least one copy of the causal variant.}

\item{FamID}{Numeric. The family ID to assign to the simulated pedigree.}

\item{founder_byears}{Numeric vector of length 2.  The span of years from which to simulate, uniformly, the birth year for the founder who introduced the rare variant to the pedigree.}

\item{stop_year}{Numeric. The last year of study.  If not supplied, defaults to the current year.}

\item{carrier_prob}{Numeric.  The carrier probability for all causal variants with relative-risk of disease \code{GRR}.  By default, \code{carrier_prob}\code{ = 0.002}}

\item{RVfounder}{Logical.  Indicates if all pedigrees segregate the rare, causal variant.  By default, \code{RVfounder = FALSE} See details.}

\item{birth_range}{Numeric vector of length 2. The minimum and maximum allowable ages, in years, between which individuals may reproduce.  By default, \code{c(18, 45)}.}

\item{NB_params}{Numeric vector of length 2. The size and probability parameters of the negative binomial distribution used to model the number of children per household.  By default, \code{NB_params}\code{ = c(2, 4/7)}, due to the investigation of Kojima and Kelleher (1962).}

\item{fert}{Numeric.  A constant used to rescale the fertility rate after disease-onset. By default, \code{fert = 1}.}
}
\value{
The simulated pedigree.
}
\description{
Please note the distinction between \code{sim_ped} and \code{sim_RVped}.  Pedigrees simulated using \code{sim_ped} do not account for study design.  To simulate a pedigree ascertained to contain multiple family members affected by a disease please use \code{\link{sim_RVped}}.
}
\details{
To introduce the rare variant to the pedigree, We allow users to choose from one of the following two assumptions:
\enumerate{
\item Assume that the variant is rare enough that a single copy has been introduced by one founder, and begin the simulation of the pedigree with this founder, as in Bureau (2014).
\item Simulate the starting founder's rare-variant status with probability equal to the carrier probability of the rare variant in the population.  We note that under this setting pedigrees may not segregate the rare variant.
}
The \code{sim_ped} function starts simulating the pedigree by generating the birth year for the starting founder, uniformly between the years specified by \code{founder_byears}.  Next, all life events are simulated for the founder via \code{\link{sim_life}}.  Possible life events include: reproduction, disease onset, and death.  We only allow disease onset to occur once, i.e. no remission.  Computationally, this implies that after disease onset, the waiting time to death is always simulated using the age-specific mortality rates for the \emph{affected} population.  Life events for individuals who have inherited the rare variant are simulated such that their relative-risk of disease is \code{GRR}, according to a proportional hazards model.  The relative-risk of disease onset for individuals who have not inherited the causal variant is assumed to be 1.  Any life events that occur after \code{stop_year} are censored.

When segregating in the pedigree, the rare variant is transmitted from parent to offspring according to Mendel's laws.  The process of simulating life events is repeated for any offspring that are produced before \code{stop_year}.
}
\section{See Also}{

\code{\link{sim_RVped}}
}

\examples{
data(AgeSpecific_Hazards)

#Simulate a random pedigree
set.seed(22)
ex_ped <- sim_ped(hazard_rates = hazard(hazardDF = AgeSpecific_Hazards),
                  GRR = 10,
                  FamID = 1,
                  founder_byears = c(1900, 1910),
                  stop_year = 2015)

ex_ped
plot(ex_ped, location = "bottomleft")
plot(ex_ped, ref_year = 2015,
     cex= 0.75, symbolsize = 1.25,
     location = "bottomleft")

summary(ex_ped)

#Simulate a random pedigree
set.seed(22)
ex_ped <- sim_ped(hazard_rates = hazard(hazardDF = AgeSpecific_Hazards),
                  RVfounder = TRUE,
                  GRR = 10,
                  FamID = 1,
                  founder_byears = c(1900, 1910),
                  stop_year = 2015)

ex_ped
plot(ex_ped)
plot(ex_ped, ref_year = 2015,
     cex= 0.75, symbolsize = 1.25)
summary(ex_ped)

}
\references{
Nieuwoudt, Christina and Jones, Samantha J and Brooks-Wilson, Angela and Graham, Jinko. (14 December 2017) \emph{Simulating Pedigrees Ascertained for Multiple Disease-Affected Relatives}. bioRxiv 234153.

Ken-Ichi Kojima, Therese M. Kelleher. (1962), \emph{Survival of Mutant Genes}. The American Naturalist 96, 329-346.

Alexandre Bureau, Samuel G. Younkin, Margaret M. Parker, Joan E. Bailey-Wilson, Mary L. Marazita, Jeffrey C. Murray, Elisabeth Mangold, Hasan Albacha-Hejazi, Terri H. Beaty, and Ingo Ruczinski (2014). \emph{Inferring rare disease risk variants based on exact probabilities of sharing by multiple affected relatives.} Bioinformatics; Vol. 30, No. 15, pp. 2189-2196.
}
