% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/smoothArea.R
\name{smoothArea}
\alias{smoothArea}
\title{Smooth via area level model}
\usage{
smoothArea(
  formula,
  domain,
  design = NULL,
  responseType = c("gaussian", "binary")[1],
  Amat = NULL,
  direct.est = NULL,
  X.area = NULL,
  domain.size = NULL,
  pc.u = 1,
  pc.alpha = 0.01,
  pc.u.phi = 0.5,
  pc.alpha.phi = 2/3,
  CI = 0.95,
  n.sample = 250,
  var.tol = 1e-10
)
}
\arguments{
\item{formula}{an object of class 'formula' describing the model to be fitted. 
If direct.est is specified, the right hand side of the formula is not necessary.}

\item{domain}{formula specifying variable containing domain labels}

\item{design}{an object of class "svydesign" containing the data for the model}

\item{responseType}{of the response variable, currently supports 'binary' (default with logit link function) or 'gaussian'.}

\item{Amat}{adjacency matrix for the regions. If set to NULL, the IID spatial effect will be used.}

\item{direct.est}{data frame of direct estimates, with first column containing domain, second column containing direct estimate, and third column containing variance of direct estimate.}

\item{X.area}{areal covariates data frame. One of the column names needs to match 'domain', in order to be linked to the data input. Currently only supporting time-invariant domain-level covariates.}

\item{domain.size}{domain size data frame. One of the column names needs to match 'domain' in order to be linked to the data input and there must be a size column containing domain sizes.}

\item{pc.u}{hyperparameter U for the PC prior on precisions.}

\item{pc.alpha}{hyperparameter alpha for the PC prior on precisions.}

\item{pc.u.phi}{hyperparameter U for the PC prior on the mixture probability phi in BYM2 model.}

\item{pc.alpha.phi}{hyperparameter alpha for the PC prior on the mixture probability phi in BYM2 model.}

\item{CI}{the desired posterior credible interval to calculate}

\item{n.sample}{number of draws from posterior used to compute summaries}

\item{var.tol}{tolerance parameter; if variance of an area's direct estimator is below this value, that direct estimator is dropped from model}
}
\value{
A list with elements
\item{direct.est}{direct estimates}
\item{s.dir.iid.fit}{fitted INLA object for iid domain effects model}
\item{s.dir.iid.est}{non-spatial smoothed estimates}
\item{s.dir.sp.fit}{fitted INLA object for spatial domain effects model}
\item{s.dir.sp.est}{spatially smoothed estimates (if adjacency matrix provided)}
}
\description{
Generates small area estimates  by smoothing direct estimates using an area 
level model
}
\examples{
\dontrun{
library(survey)
data(DemoData2)
data(DemoMap2)
des0 <- svydesign(ids = ~clustid+id, strata = ~strata,
                  weights = ~weights, data = DemoData2, nest = T)
Xmat <- aggregate(age~region, data = DemoData2, FUN = mean)

EXAMPLE 1: Continuous response model
cts.res <- smoothArea(tobacco.use ~ 1, domain = ~region,
                      Amat = DemoMap2$Amat, design = des0,
                      pc.u = 1,
                      pc.alpha = 0.01,
                      pc.u.phi = 0.5,
                      pc.alpha.phi = 2/3)
                      
EXAMPLE 2: Including area level covariates
cts.cov.res <- smoothArea(tobacco.use ~ age, domain = ~region,
                          Amat = DemoMap2$Amat, design = des0,
                          X.area = Xmat,
                          pc.u = 1,
                          pc.alpha = 0.01,
                          pc.u.phi = 0.5,
                          pc.alpha.phi = 2/3)
                          
EXAMPLE 3: Binary response model
bin.res <- smoothArea(tobacco.use ~ 1, domain = ~region,
                      responseType = "binary",
                      Amat = DemoMap2$Amat, design = des0,
                      pc.u = 1,
                      pc.alpha = 0.01,
                      pc.u.phi = 0.5,
                      pc.alpha.phi = 2/3)
                      
EXAMPLE 4: Including area level covariates in binary response model
bin.cov.res <- smoothArea(tobacco.use ~ age, domain = ~region,
                          responseType = "binary",
                          Amat = DemoMap2$Amat, design = des0,
                          X.area = Xmat,
                          pc.u = 1,
                          pc.alpha = 0.01,
                          pc.u.phi = 0.5,
                          pc.alpha.phi = 2/3)
}

}
