% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/optimalPortfolio.R
\name{optimalPortfolio}
\alias{optimalPortfolio}
\title{Optimal portfolio}
\usage{
optimalPortfolio(Sigma, mu = NULL, semiDev = NULL, control = list())
}
\arguments{
\item{Sigma}{a \eqn{(N \times N)}{(N x N)} covariance matrix.}

\item{mu}{a \eqn{(N \times 1)}{(N x 1)} vector of expected returns. Default:
\code{mu = NULL}.}

\item{semiDev}{a vector \eqn{(N \times 1)}{(N x 1)} of semideviations.
Default: \code{semiDev = NULL}.}

\item{control}{control parameters (see *Details*).}
}
\value{
A \eqn{(N \times 1)}{(N x 1)} vector of optimal portfolio weights.
}
\description{
Function wich computes the optimal portfolio's weights.
}
\details{
The argument \code{control} is a list that can supply any of the following
components: 
\itemize{
\item \code{type} method used to compute the
optimal portfolio, among \code{'mv'}, \code{'minvol'}, \code{'invvol'},
\code{'erc'}, \code{'maxdiv'}, \code{'riskeff'} and \code{'maxdec'} where: 

\code{'mv'} is used to compute the weights of the mean-variance portfolio. The weights are
computed following this equation: \deqn{w = \frac{1}{\gamma} \Sigma^{-1}
\mu }{w = 1 / \gamma \Sigma^{-1} \mu}. 

\code{'minvol'} is used to compute the weights of the minimum variance portfolio.  

\code{'invvol'} is the inverse volatility portfolio.

\code{'erc'} is used to compute the weights of the equal-risk-contribution portfolio. For a 
portfolio \eqn{w}, the percentage volatility risk contribution of the i-th
asset in the portfolio is given by: 
\deqn{\% RC_i = \frac{ w_i {[\Sigma w]}_i}{w' \Sigma w} }{ RC_i = w_i[\Sigma w]_i / (w' \Sigma w)}. 
Then we compute the optimal portfolio by solving the following optimization problem:
\deqn{w = argmin \left\{ \sum_{i=1}^N (\% RC_i - \frac{1}{N})^2 \right\}
}{ argmin { \sum_{i=1}^{N} (RC_i - 1/N)^2} }.

\code{'maxdiv'} is used to compute the weights of the maximum diversification portfolio where:
\deqn{DR(w) = \frac{ w' \sigma}{\sqrt{w' \Sigma w} } \geq 1 }{ DR(w) = (w'
\sigma)/(\sqrt(w' \Sigma w)) \ge 1} is used in the optimization problem.

\code{'riskeff'} is used to compute the weights of the risk-efficient
portfolio: \deqn{w = {argmax}\left\{ \frac{w' J \xi}{ \sqrt{w' \Sigma w}
}\right\} }{w = argmax ( w'J \xi)\sqrt(w'\Sigma w)} where \eqn{J} is a
\eqn{(N \times 10)}{(N x 10)} matrix of zeros whose \eqn{(i,j)}-th element
is one if the semi-deviation of stock \eqn{i} belongs to decile
\eqn{j},\eqn{\xi = (\xi_1,\ldots,\xi_{10})'}. 

\code{'maxdec'} is used to compute the weights of the maximum-decorrelation
portfolio: \deqn{w = {argmax}\left\{ 1 -  \sqrt{w' \Sigma w} \right\}
}{w = argmax {1- \sqrt(w' R w)}} where \eqn{R} is the correlation matrix. 

Default: \code{type = 'mv'}.

These portfolios are summarized in Ardia and Boudt (2015) and Ardia et al. (2017). Below we list the various references.

\item \code{constraint} constraint used for the optimization, among
\code{'none'}, \code{'lo'}, \code{'gross'} and \code{'user'}, where: \code{'none'} is used to 
compute the unconstraint portfolio, \code{'lo'} is the long-only constraints (non-negative weighted),  
\code{'gross'} is the gross exposure constraint, and \code{'user'} is the set of user constraints (typically
lower and upper boundaries. Default: \code{constraint = 'none'}. Note that the 
summability constraint is always imposed.

\item \code{LB} lower boundary for the weights. Default: \code{LB = NULL}. 

\item \code{UB} lower boundary for the weights. Default: \code{UB = NULL}. 

\item \code{w0} starting value for the optimizer. Default: \code{w0 = NULL} takes the 
equally-weighted portfolio as a starting value. When \code{LB} and \code{UB} are provided, it is set to 
mid-point of the bounds.

\item \code{gross.c} gross exposure constraint. Default: \code{gross.c = 1.6}. 

\item \code{gamma} risk aversion parameter. Default: \code{gamma = 0.89}.

\item \code{ctr.slsqp} list with control parameters for slsqp function.
}
}
\examples{
# Load returns of assets or portfolios
data("Industry_10")
rets = Industry_10

# Mean estimation
mu = meanEstimation(rets)

# Covariance estimation
Sigma = covEstimation(rets)

# Semi-deviation estimation
semiDev = semidevEstimation(rets)

# Mean-variance portfolio without constraint and gamma = 0.89
optimalPortfolio(mu = mu, Sigma = Sigma)

# Mean-variance portfolio without constraint and gamma = 1
optimalPortfolio(mu = mu, Sigma = Sigma, 
  control = list(gamma = 1))

# Mean-variance portfolio without constraint and gamma = 0.89
optimalPortfolio(mu = mu, Sigma = Sigma, 
  control = list(type = 'mv'))

# Mean-variance portfolio without constraint and gamma = 0.89
optimalPortfolio(mu = mu, Sigma = Sigma, 
  control = list(type = 'mv', constraint = 'none'))

# Mean-variance portfolio with the long-only constraint and gamma = 0.89
optimalPortfolio(mu = mu, Sigma = Sigma, 
  control = list(type = 'mv', constraint = 'lo'))

# Mean-variance portfolio with LB and UB constraints
optimalPortfolio(mu = mu, Sigma = Sigma, 
  control = list(type = 'mv', constraint = 'user', LB = rep(0.02, 10), UB = rep(0.8, 10)))

# Mean-variance portfolio with the gross constraint, 
# gross constraint parameter = 1.6 and gamma = 0.89
optimalPortfolio(mu = mu, Sigma = Sigma, 
  control = list(type = 'mv', constraint = 'gross'))

# Mean-variance portfolio with the gross constraint, 
# gross constraint parameter = 1.2 and gamma = 0.89
optimalPortfolio(mu = mu, Sigma = Sigma, 
  control = list(type = 'mv', constraint = 'gross', gross.c = 1.2))

# Minimum volatility portfolio without constraint
optimalPortfolio(Sigma = Sigma, 
  control = list(type = 'minvol'))

# Minimum volatility portfolio without constraint
optimalPortfolio(Sigma = Sigma, 
  control = list(type = 'minvol', constraint = 'none'))

# Minimim volatility portfolio with the long-only constraint
optimalPortfolio(Sigma = Sigma, 
  control = list(type = 'minvol', constraint = 'lo'))
  
# Minimim volatility portfolio with LB and UB constraints
optimalPortfolio(Sigma = Sigma, 
  control = list(type = 'minvol', constraint = 'user', LB = rep(0.02, 10), UB = rep(0.8, 10)))

# Minimum volatility portfolio with the gross constraint 
# and the gross constraint parameter = 1.6
optimalPortfolio(Sigma = Sigma, 
  control = list(type = 'minvol', constraint = 'gross'))

# Minimum volatility portfolio with the gross constraint 
# and the gross parameter = 1.2
optimalPortfolio(Sigma = Sigma, 
  control = list(type = 'minvol', constraint = 'gross', gross.c = 1.2))
    
# Inverse volatility portfolio
optimalPortfolio(Sigma = Sigma, 
  control = list(type = 'invvol'))

# Equal-risk-contribution portfolio with the long-only constraint
optimalPortfolio(Sigma = Sigma, 
  control = list(type = 'erc', constraint = 'lo'))
  
# Equal-risk-contribution portfolio with LB and UB constraints
optimalPortfolio(Sigma = Sigma, 
  control = list(type = 'erc', constraint = 'user', LB = rep(0.02, 10), UB = rep(0.8, 10)))

# Maximum diversification portfolio without constraint
optimalPortfolio(Sigma = Sigma, 
  control = list(type = 'maxdiv'))

# Maximum diversification portoflio with the long-only constraint
optimalPortfolio(Sigma = Sigma, 
  control = list(type = 'maxdiv', constraint = 'lo'))
  
# Maximum diversification portoflio with LB and UB constraints
optimalPortfolio(Sigma = Sigma, 
  control = list(type = 'maxdiv', constraint = 'user', LB = rep(0.02, 10), UB = rep(0.8, 10)))

# Risk-efficient portfolio without constraint
optimalPortfolio(Sigma = Sigma, semiDev = semiDev, 
  control = list(type = 'riskeff'))

# Risk-efficient portfolio with the long-only constraint
optimalPortfolio(Sigma = Sigma, semiDev = semiDev, 
  control = list(type = 'riskeff', constraint = 'lo'))
  
# Risk-efficient portfolio with LB and UB constraints
optimalPortfolio(Sigma = Sigma, semiDev = semiDev, 
  control = list(type = 'riskeff', constraint = 'user', LB = rep(0.02, 10), UB = rep(0.8, 10)))
  
# Maximum decorrelation portfolio without constraint
optimalPortfolio(Sigma = Sigma, 
  control = list(type = 'maxdec'))

# Maximum decorrelation portoflio with the long-only constraint
optimalPortfolio(Sigma = Sigma, 
  control = list(type = 'maxdec', constraint = 'lo'))
  
# Maximum decorrelation portoflio with LB and UB constraints
optimalPortfolio(Sigma = Sigma, 
  control = list(type = 'maxdec', constraint = 'user', LB = rep(0.02, 10), UB = rep(0.8, 10)))
}
\references{
Amenc, N., Goltz, F., Martellini, L., Retowsky, P. (2011).
Efficient indexation: An alternatice to cap-weightes indices.  
\emph{Journal of Investment Management} \bold{9}(4), pp.1-23.

Ardia, D., Boudt, K. (2015). 
Implied expected returns and the choice of a mean-variance efficient portfolio proxy.
\emph{Journal of Portfolio Management} \bold{41}(4), pp.66-81. 
\doi{10.3905/jpm.2015.41.4.068}

Ardia, D., Bolliger, G., Boudt, K., Gagnon-Fleury, J.-P. (2017).  
The Impact of covariance misspecification in risk-based portfolios.  
\emph{Annals of Operations Research} \bold{254}(1-2), pp.1-16. 
\doi{10.1007/s10479-017-2474-7}

Choueifaty, Y., Coignard, Y. (2008).  
Toward maximum diversification.
\emph{Journal of Portfolio Management} \bold{35}(1), pp.40-51. 
\doi{10.3905/JPM.2008.35.1.40}

Choueifaty, Y., Froidure, T., Reynier, J. (2013).  
Properties of the most diversified portfolio.  
\emph{Journal of Investment Strategies} \bold{2}(2), pp.49-70. 
\doi{10.21314/JOIS.2013.033}

Das, S., Markowitz, H., Scheid, J., Statman, M. (2010).  
Portfolio optimization with mental accounts.  
\emph{Journal of Financial and Quantitative Analysis} \bold{45}(2), pp.311-334. 
\doi{10.1017/S0022109010000141}

DeMiguel, V., Garlappi, L., Uppal, R. (2009).  
Optimal versus naive diversification: How inefficient is the 1/n portfolio strategy.  
\emph{Review of Financial Studies} \bold{22}(5), pp.1915-1953. 
\doi{10.1093/rfs/hhm075}

Fan, J., Zhang, J., Yu, K. (2012).  
Vast portfolio selection with gross-exposure constraints.
\emph{Journal of the American Statistical Association} \bold{107}(498), pp.592-606. 
\url{10.1080/01621459.2012.68282}

Maillard, S., Roncalli, T., Teiletche, J. (2010).  
The properties of equally weighted risk contribution portfolios.  
\emph{Journal of Portfolio Management} \bold{36}(4), pp.60-70. 
\doi{10.3905/jpm.2010.36.4.060}

Martellini, L. (2008).  
Towards the design of better equity benchmarks.
\emph{Journal of Portfolio Management} \bold{34}(4), Summer,pp.34-41. 
\doi{10.3905/jpm.2008.709978}
}
\author{
David Ardia, Kris Boudt and Jean-Philippe Gagnon Fleury.
}
\keyword{optimize}
