% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predictionInterval.R
\name{predictionInterval}
\alias{predictionInterval}
\title{Prediction interval for effect estimate of replication study}
\usage{
predictionInterval(
  thetao,
  seo,
  ser,
  tau = 0,
  conf.level = 0.95,
  designPrior = "predictive"
)
}
\arguments{
\item{thetao}{Numeric vector of effect estimates from original studies.}

\item{seo}{Numeric vector of standard errors of the original effect estimates.}

\item{ser}{Numeric vector of standard errors of the replication effect estimates.}

\item{tau}{Between-study heterogeneity standard error.
Default is \code{0} (no heterogeneity).
Is only taken into account when \code{designPrior} is "predictive" or "EB".}

\item{conf.level}{The confidence level of the prediction intervals. Default is 0.95.}

\item{designPrior}{Either "predictive" (default), "conditional", or "EB".
If "EB", the contribution of the original study to the predictive distribution is
shrunken towards zero based on the evidence in the original study (with empirical Bayes).}
}
\value{
A data frame with the following columns
\item{lower}{Lower limit of prediction interval,}
\item{mean}{Mean of predictive distribution,}
\item{upper}{Upper limit of prediction interval.}
}
\description{
Computes a prediction interval for the effect estimate of the replication study.
}
\details{
This function computes a prediction interval and a mean estimate under a
specified predictive distribution of the replication effect estimate. Setting
\code{designPrior = "conditional"} is not recommended since this ignores the
uncertainty of the original effect estimate. See Patil, Peng, and Leek (2016)
and Pawel and Held (2020) for details.

\code{predictionInterval} is the vectorized version of \code{.predictionInterval_}.
\code{\link[base]{Vectorize}} is used to vectorize the function.
}
\examples{
predictionInterval(thetao = c(1.5, 2, 5), seo = 1, ser = 0.5, designPrior = "EB")

# compute prediction intervals for replication projects
data("RProjects", package = "ReplicationSuccess")
parOld <- par(mfrow = c(2, 2))
for (p in unique(RProjects$project)) {
  data_project <- subset(RProjects, project == p)
  PI <- predictionInterval(thetao = data_project$fiso, seo = data_project$se_fiso,
                           ser = data_project$se_fisr)
  PI <- tanh(PI) # transforming back to correlation scale
  within <- (data_project$rr < PI$upper) & (data_project$rr > PI$lower)
  coverage <- mean(within)
  color <- ifelse(within == TRUE, "#333333B3", "#8B0000B3")
  study <- seq(1, nrow(data_project))
  plot(data_project$rr, study, col = color, pch = 20,
       xlim = c(-0.5, 1), xlab = expression(italic(r)[r]),
       main = paste0(p, ": ", round(coverage*100, 1), "\% coverage"))
  arrows(PI$lower, study, PI$upper, study, length = 0.02, angle = 90,
         code = 3, col = color)
  abline(v = 0, lty = 3)
}
par(parOld)
}
\references{
Patil, P., Peng, R. D., Leek, J. T. (2016).
What should researchers expect when they replicate studies? A statistical view of
replicability in psychological science. \emph{Perspectives on Psychological Science},
\bold{11}, 539-544.  \doi{10.1177/1745691616646366}

Pawel, S., Held, L. (2020). Probabilistic forecasting of replication studies.
\emph{PLoS ONE}. \bold{15}, e0231416. \doi{10.1371/journal.pone.0231416}
}
\author{
Samuel Pawel
}
