% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/exactRLRT.R
\name{exactRLRT}
\alias{exactRLRT}
\title{Restricted Likelihood Ratio Tests for additive and linear mixed models}
\usage{
exactRLRT(m, mA = NULL, m0 = NULL, seed = NA, nsim = 10000,
  log.grid.hi = 8, log.grid.lo = -10, gridlength = 200,
  parallel = c("no", "multicore", "snow"), ncpus = 1L, cl = NULL)
}
\arguments{
\item{m}{The fitted model under the alternative or, for testing in models
with multiple variance components, the reduced model containing only the
random effect to be tested (see Details), an \code{lme}, \code{lmerMod} or
\code{spm} object}

\item{mA}{The full model under the alternative for testing in models with
multiple variance components}

\item{m0}{The model under the null for testing in models with multiple
variance components}

\item{seed}{input for \code{set.seed}}

\item{nsim}{Number of values to simulate}

\item{log.grid.hi}{Lower value of the grid on the log scale. See
\code{\link{exactRLRT}}.}

\item{log.grid.lo}{Lower value of the grid on the log scale. See
\code{\link{exactRLRT}}.}

\item{gridlength}{Length of the grid. See \code{\link{exactLRT}}.}

\item{parallel}{The type of parallel operation to be used (if any). If
missing, the default is "no parallelization").}

\item{ncpus}{integer: number of processes to be used in parallel operation:
typically one would chose this to the number of available CPUs. Defaults to
1, i.e., no parallelization.}

\item{cl}{An optional parallel or snow cluster for use if parallel = "snow".
If not supplied, a cluster on the local machine is created for the duration
of the call.}
}
\value{
A list of class \code{htest} containing the following components:

A list of class \code{htest} containing the following components:
\itemize{
\item \code{statistic} the observed likelihood ratio
\item \code{p} p-value for the observed test statistic
\item \code{method} a character string indicating what type of test was
performed and how many values were simulated to determine the critical value
\item \code{sample} the samples from the null distribution returned by
\code{\link{RLRTSim}}
}
}
\description{
This function provides an (exact) restricted likelihood ratio test based on
simulated values from the finite sample distribution for testing whether the
variance of a random effect is 0 in a linear mixed model with known
correlation structure of the tested random effect and i.i.d. errors.
}
\details{
Testing in models with only a single variance component require only the
first argument \code{m}. For testing in models with multiple variance
components, the fitted model \code{m} must contain \bold{only} the random
effect set to zero under the null hypothesis, while \code{mA} and \code{m0}
are the models under the alternative and the null, respectively. For models
with a single variance component, the simulated distribution is exact if the
number of parameters (fixed and random) is smaller than the number of
observations. Extensive simulation studies (see second reference below)
confirm that the application of the test to models with multiple variance
components is safe and the simulated distribution is correct as long as the
number of parameters (fixed and random) is smaller than the number of
observations and the nuisance variance components are not superfluous or
very small. We use the finite sample distribution of the restricted
likelihood ratio test statistic as derived by Crainiceanu & Ruppert (2004).
}
\examples{

library(lme4)
data(sleepstudy)
mA <- lmer(Reaction ~ I(Days-4.5) + (1|Subject) + (0 + I(Days-4.5)|Subject), 
  data = sleepstudy)
m0 <- update(mA, . ~ . - (0 + I(Days-4.5)|Subject))
m.slope  <- update(mA, . ~ . - (1|Subject))
#test for subject specific slopes:
exactRLRT(m.slope, mA, m0)

library(mgcv)
data(trees)
#test quadratic trend vs. smooth alternative
m.q<-gamm(I(log(Volume)) ~ Height + s(Girth, m = 3), data = trees, 
  method = "REML")$lme
exactRLRT(m.q)
#test linear trend vs. smooth alternative
m.l<-gamm(I(log(Volume)) ~ Height + s(Girth, m = 2), data = trees, 
  method = "REML")$lme
exactRLRT(m.l)

}
\author{
Fabian Scheipl, bug fixes by Andrzej Galecki, updates for
\pkg{lme4}-compatibility by Ben Bolker
}
\references{
Crainiceanu, C. and Ruppert, D. (2004) Likelihood ratio tests in
linear mixed models with one variance component, \emph{Journal of the Royal
Statistical Society: Series B},\bold{66},165--185.

Greven, S., Crainiceanu, C., Kuechenhoff, H., and Peters, A. (2008)
Restricted Likelihood Ratio Testing for Zero Variance Components in Linear
Mixed Models, \emph{Journal of Computational and Graphical Statistics},
\bold{17} (4): 870--891.

Scheipl, F., Greven, S. and Kuechenhoff, H. (2008) Size and power of tests
for a zero random effect variance or polynomial regression in additive and
linear mixed models.  \emph{Computational Statistics & Data Analysis},
\bold{52}(7):3283--3299.
}
\seealso{
\code{\link{RLRTSim}} for the underlying simulation algorithm;
\code{\link{exactLRT}} for likelihood based tests
}
\keyword{htest}

