% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CalculateNonLinearParameters.R
\name{CalculateEmbeddingDim}
\alias{CalculateEmbeddingDim}
\title{Estimate the proper embedding dimension for the RR time series}
\usage{
CalculateEmbeddingDim(HRVData, numberPoints = 5000, timeLag = 1,
  maxEmbeddingDim = 15, threshold = 0.95, maxRelativeChange = 0.05,
  doPlot = TRUE)
}
\arguments{
\item{HRVData}{Data structure that stores the beats register and information related to it}

\item{numberPoints}{Number of points from the time series that will be used to estimate
the embedding dimension. By default, 5000 points are used.}

\item{timeLag}{Time lag used to build the Takens' vectors needed to estimate the
embedding dimension (see \link{buildTakens}). Default: 1.}

\item{maxEmbeddingDim}{Maximum possible embedding dimension for the time series. Default: 15.}

\item{threshold}{Numerical value between 0 and 1. The embedding dimension is
estimated using the E1(d) function. E1(d) stops changing when d is greater 
than or equal to embedding dimension, staying close to 1. This value 
establishes a threshold for considering that E1(d) is close to 1.
Default: 0.95}

\item{maxRelativeChange}{Maximum relative change in E1(d) with respect to
E1(d-1) in order to consider that the E1 function has been stabilized and it
will stop changing. Default: 0.05.}

\item{doPlot}{Logical value. If TRUE (default value), a plot of E1(d) and E2(d) is shown.}
}
\description{
This function determines the minimum embedding dimension from a scalar time 
series using the algorithm proposed by L. Cao (see references).
}
\details{
The Cao's algorithm uses 2 functions in order to estimate the embedding dimension
from a time series: the E1(d) and the E2(d) functions, where d denotes the dimension.

E1(d) stops changing when d is greater than or equal to the embedding dimension, staying close to 1.
On the other hand, E2(d) is used to distinguish deterministic signals from stochastic signals. For 
deterministic signals, there exists some d such that E2(d)!=1. For stochastic signals,
E2(d) is approximately 1 for all the values.
}
\note{
The current implementation of this function is fully written in R, based on the 
\code{\link[nonlinearTseries]{estimateEmbeddingDim}} function from the 
nonlinearTseries package. Thus it requires 
heavy computations and may be quite slow. The \emph{numberPoints} parameter can be used
for controlling the computational burden.

Future versions of the package will solve this issue.
}
\examples{
\dontrun{
data(HRVProcessedData)
HRVData = HRVProcessedData
HRVData = SetVerbose(HRVData,T)
timeLag = CalculateTimeLag(HRVData,technique = "ami")
embeddingDim = CalculateEmbeddingDim(HRVData,
                                     timeLag = timeLag,
                                     maxEmbeddingDim = 15)
}
}
\references{
Cao, L. Practical method for determining the minimum embedding dimension of a scalar time series. Physica D: Nonlinear Phenomena,
110,1, pp. 43-50 (1997).
}
\seealso{
\code{\link[nonlinearTseries]{estimateEmbeddingDim}}.
}

