% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/redcap-read-oneshot.R
\name{redcap_read_oneshot}
\alias{redcap_read_oneshot}
\title{Read/Export records from a REDCap project.}
\usage{
redcap_read_oneshot(redcap_uri, token, records = NULL,
  records_collapsed = "", fields = NULL, fields_collapsed = "",
  events = NULL, events_collapsed = "", export_data_access_groups = FALSE,
  filter_logic = "", raw_or_label = "raw", verbose = TRUE,
  config_options = NULL)
}
\arguments{
\item{redcap_uri}{The URI (uniform resource identifier) of the REDCap project.  Required.}

\item{token}{The user-specific string that serves as the password for a project.  Required.}

\item{records}{An array, where each element corresponds to the ID of a desired record.  Optional.}

\item{records_collapsed}{A single string, where the desired ID values are separated by commas.  Optional.}

\item{fields}{An array, where each element corresponds a desired project field.  Optional.}

\item{fields_collapsed}{A single string, where the desired field names are separated by commas.  Optional.}

\item{events}{An array, where each element corresponds a desired project event  Optional.}

\item{events_collapsed}{A single string, where the desired event names are separated by commas.  Optional.}

\item{export_data_access_groups}{A boolean value that specifies whether or not to export the ``redcap_data_access_group'' field when data access groups are utilized in the project. Default is \code{FALSE}. See the details below.}

\item{filter_logic}{String of logic text (e.g., \code{[gender] = 'male'}) for filtering the data to be returned by this API method, in which the API will only return the records (or record-events, if a longitudinal project) where the logic evaluates as TRUE.   An blank/empty string returns all records.}

\item{raw_or_label}{A string (either \code{'raw'} or \code{'label'} that specifies whether to export the raw coded values or the labels for the options of multiple choice fields.  Default is \code{'raw'}.}

\item{verbose}{A boolean value indicating if \code{message}s should be printed to the R console during the operation.  The verbose output might contain sensitive information (\emph{e.g.} PHI), so turn this off if the output might be visible somewhere public. Optional.}

\item{config_options}{A list of options to pass to \code{POST} method in the \code{httr} package.  See the details below. Optional.}
}
\value{
Currently, a list is returned with the following elements,
\enumerate{
 \item \code{data}: An R \code{data.frame} of the desired records and columns.
 \item \code{success}: A boolean value indicating if the operation was apparently successful.
 \item \code{status_code}: The \href{http://en.wikipedia.org/wiki/List_of_HTTP_status_codes}{http status code} of the operation.
 \item \code{outcome_message}: A human readable string indicating the operation's outcome.
 \item \code{records_collapsed}: The desired records IDs, collapsed into a single string, separated by commas.
 \item \code{fields_collapsed}: The desired field names, collapsed into a single string, separated by commas.
 \item \code{filter_logic}: The filter statement passed as an argument.
 \item \code{elapsed_seconds}: The duration of the function.
 \item \code{raw_text}: If an operation is NOT successful, the text returned by REDCap.  If an operation is successful, the `raw_text` is returned as an empty string to save RAM.
}
}
\description{
This function uses REDCap's API to select and return data.
}
\details{
The full list of configuration options accepted by the \code{httr} package is viewable by executing \code{httr::httr_options()}.  The \code{httr}
package and documentation is available at \url{https://cran.r-project.org/package=httr}.

If you do not pass in this export_data_access_groups value, it will default to \code{FALSE}. The following is from the API help page for version 5.2.3: This flag is only viable if the user whose token is being used to make the API request is *not* in a data access group. If the user is in a group, then this flag will revert to its default value.
}
\examples{
\dontrun{
library(REDCapR) #Load the package into the current R session.
uri      <- "https://bbmc.ouhsc.edu/redcap/api/"
token    <- "9A81268476645C4E5F03428B8AC3AA7B"
#Return all records and all variables.
ds_all_rows_all_fields <- redcap_read_oneshot(redcap_uri=uri, token=token)$data

#Return only records with IDs of 1 and 3
desired_records_v1 <- c(1, 3)
ds_some_rows_v1 <- redcap_read_oneshot(
   redcap_uri = uri,
   token      = token,
   records    = desired_records_v1
)$data

#Return only the fields record_id, name_first, and age
desired_fields_v1 <- c("record_id", "name_first", "age")
ds_some_fields_v1 <- redcap_read_oneshot(
   redcap_uri = uri,
   token      = token,
   fields     = desired_fields_v1
)$data


#Use the SSL cert file that come with the openssl package.
cert_location <- system.file("cacert.pem", package="openssl")
if( file.exists(cert_location) ) {
  config_options         <- list(cainfo=cert_location)
  ds_different_cert_file <- redcap_read_oneshot(
    redcap_uri     = uri,
    token          = token,
    config_options = config_options
  )$data
}

#Force the connection to use SSL=3 (which is not preferred, and possibly insecure).
config_options <- list(sslversion=3)
ds_ssl_3 <- redcap_read_oneshot(
  redcap_uri     = uri,
  token          = token,
  config_options = config_options
)$data

config_options <- list(ssl.verifypeer=FALSE)
ds_no_ssl <- redcap_read_oneshot(
   redcap_uri     = uri,
   token          = token,
   config_options = config_options
)$data
}
}
\author{
Will Beasley
}
\references{
The official documentation can be found on the `API Help Page` and `API Examples' pages
on the REDCap wiki (ie, \url{https://community.projectredcap.org/articles/456/api-documentation.html} and
\url{https://community.projectredcap.org/articles/462/api-examples.html}). If you do not have an account
for the wiki, please ask your campus REDCap administrator to send you the static material.

The official \href{http://curl.haxx.se}{cURL site} discusses the process of using SSL to verify the server being connected to.
}

