% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/check_queries.R
\name{check_queries}
\alias{check_queries}
\title{Check for Changes Between Two Query Reports}
\usage{
check_queries(old, new, report_title = NULL, return_viewer = TRUE)
}
\arguments{
\item{old}{Data frame containing the previous (older) query report. Must include
\code{Identifier}, \code{Description} and \code{Query} columns (character or factor).}

\item{new}{Data frame containing the newer query report. Must include
\code{Identifier}, \code{Description} and \code{Query} columns (character or factor).
If \code{new} contains a \code{Code} column, it will be removed at the start of processing.}

\item{report_title}{Optional single string used as the caption for the HTML summary table.
Defaults to \code{"Comparison report"} when not supplied or when \code{NA}.}

\item{return_viewer}{Logical; if \code{TRUE} (default) an HTML table (knitr/kable + kableExtra)
summarizing the counts per state is produced and returned in the \code{results} element of the
returned list. If \code{FALSE}, no HTML viewer is produced (useful for non-interactive runs).}
}
\value{
A list with two elements:
\describe{
\item{\code{queries}}{A data frame containing all queries present in either \code{old} or \code{new}.
A factor column \code{Modification} indicates the state for each row (levels: \code{Pending},
\code{Solved}, \code{Miscorrected}, \code{New}). The function also reassigns \code{Code}
values so codes are consistent per \code{Identifier}.}
\item{\code{results}}{If \code{return_viewer = TRUE}, an HTML \code{knitr::kable} (styled with
\code{kableExtra}) summarising totals per state. If \code{return_viewer = FALSE}, this is \code{NULL}.}
}
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}

Compare an older query report (\code{old}) with a newer one (\code{new}) and classify each
query into one of four statuses:
\itemize{
\item \strong{Pending} — the same query is present in both reports (no change detected),
\item \strong{Solved} — the query was present in the old report but is absent from the new report,
\item \strong{New} — the query appears in the new report but was not present in the old report,
\item \strong{Miscorrected} — a special case where a query in the new report is marked as \code{New}
but shares the same \code{Identifier} and \code{Description} as an existing record (suggesting a
re-issued or modified query for the same identifier).
}

The function returns a detailed comparison table of queries with a \code{Modification} factor (one of the four statuses) and an HTML summary table showing counts per status.
}
\details{
Requirements:
\itemize{
\item Both \code{old} and \code{new} must be data frames.
\item Both data frames must contain at least the following character columns:
\code{Identifier}, \code{Description}, and \code{Query}.
\item A \code{Code} column is optional; if present it will be preserved and considered
for sorting and output. If \code{Code} exists in \code{new}, it is removed at the
beginning of the routine to avoid conflicts with re-assigned codes.
}

The function merges the two reports, constructs composite keys used for comparison, classifies each row into a modification state, detects and re-labels \code{Miscorrected} cases, reassigns a \code{Code} per \code{Identifier} to keep codes consistent, and returns a detailed dataset plus an optional HTML summary viewer.
}
\section{Notes and edge cases}{

\itemize{
\item \strong{Column types:} If \code{Identifier}, \code{Description} or \code{Query} are
factors, they will be used in the comparison — it is recommended to convert them to
character prior to calling \code{check_queries()} to avoid factor-level mismatches.
\item \strong{Sorting:} When \code{Identifier} values contain a dash (e.g. \code{"100-20"}),
the function attempts to split into numeric \code{center} and \code{id} parts for
logical ordering. Otherwise \code{Identifier} is coerced to numeric for ordering.
\item \strong{Miscorrected detection:} A \code{Miscorrected} label is assigned when more
than one row shares the same \code{Identifier + Description} composite and a row is
otherwise classified as \code{New} — this signals a likely re-issued or modified query
for an existing identifier.
}
}

\examples{
# Minimal reproducible example
old <- data.frame(
  Identifier = c("100-1", "100-2", "200-1"),
  Description = c("age check", "weight check", "lab miss"),
  Query = c("is.na(age)", "is.na(weight)", "missing lab result"),
  Code = c("100-1-1", "100-2-1", "200-1-1"),
  stringsAsFactors = FALSE
)

new <- data.frame(
  Identifier = c("100-1", "200-1", "300-1"),
  Description = c("age check", "lab miss", "new query"),
  Query = c("is.na(age)", "missing lab result (clarify)", "is.na(x)"),
  stringsAsFactors = FALSE
)

res <- check_queries(old = old, new = new, report_title = "My Query Comparison")
# detailed table
head(res$queries)
# HTML summary (if in an RMarkdown or interactive viewer)
res$results

}
