% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/object_relational-Repository.R
\name{Repository}
\alias{Repository}
\alias{AbstractRepository}
\title{Repository Pattern}
\description{
Mediates between the domain and data mapping layers using a
collection-like interface for accessing domain objects.
}
\details{
\if{html}{\out{
<!-- One line about what the function does -->
}}


With \emph{\strong{Repository}}, in-memory objects do not need to know whether
there is a database present or absent, they need no SQL interface code,
and certainly no knowledge of the database schema.
\subsection{How It Works}{
\itemize{
\item \strong{Repository} isolates domain objects from details of the database
access code;
\item \strong{Repository} concentrates code of query construction; and
\item \strong{Repository} helps to minimize duplicate query logic.
}

In R, the simplest form of \strong{Repository} encapsulates \code{data.frame}
entries persisted in a data store and the operations performed over
them, providing a more object-oriented view of the persistence layer.
From the caller point of view, the location (locally or remotely), the
technology and the interface of the data store are obscured.
}

\subsection{When to Use It}{
\itemize{
\item In situations with multiple data sources.
\item In situations where the real data store, the one that is used in
production, is remote. This allows you to implement a \strong{Repository}
mock with identical queries that runs locally. Then, the mock could
be used during development and testing. The mock itself may comprise
a sample of the real data store or just fake data.
\item In situations where the real data store doesn’t exist. Implementing
a mock \strong{Repository} allows you to defer immature decisions about
the database technology and/or defer its deployment. In this way,
the temporary solution allows you to focus the development effort on
the core functionality of the application.
\item In situations where using SQL queries can be represented by
meaningful names. For example
\verb{Repository$get_efficient_cars() = SELECT * FROM mtcars WHERE mpg > 20}
\item When building \href{https://www.oreilly.com/library/view/software-architects-handbook/9781788624060/c47a09b6-91f9-4322-a6d4-9bc1604b1bdf.xhtml}{stateless microservices}.
}
}
}
\examples{
# See more examples at <https://tidylab.github.io/R6P/articles>

# The following implementation is a Repository of car models with their
# specifications.

# First, we define the class constructor, initialize, to establish a
# transient data storage.

# In this case we use a dictionary from the collections package
# <https://randy3k.github.io/collections/reference/dict.html>

# Second, we define the add, del and get functions that operate on the dictionary.

# As an optional step, we define the NULL object. In this case, rather then
# the reserved word NULL, the NULL object is a data.frame with 0 rows and
# predefined column.

TransientRepository <- R6::R6Class(
    classname = "Repository", inherit = R6P::AbstractRepository, public = list(
        initialize = function() {private$cars <- collections::dict()},
        add = function(key, value){private$cars$set(key, value); invisible(self)},
        del = function(key){private$cars$remove(key); invisible(self)},
        get = function(key){return(private$cars$get(key, default = private$NULL_car))}
    ), private = list(
        NULL_car = cbind(uid  = NA_character_, datasets::mtcars)[0,],
        cars = NULL
))

# Adding customised operations is also possible via the R6 set function.
# The following example, adds a query that returns all the objects in the database

TransientRepository$set("public", "get_all_cars", overwrite = TRUE, function(){
    result <- private$cars$values() \%>\% dplyr::bind_rows()
    if(nrow(result) == 0) return(private$NULL_car) else return(result)
})

# In this example, we use the mtcars dataset with a uid column that uniquely
# identifies the different cars in the Repository:
mtcars <- datasets::mtcars \%>\% tibble::rownames_to_column("uid")
head(mtcars, 2)

# Here is how the caller uses the Repository:

## Instantiate a repository object
repository <- TransientRepository$new()

## Add two different cars specification to the repository
repository$add(key = "Mazda RX4", value = dplyr::filter(mtcars, uid == "Mazda RX4"))
repository$add(key = "Mazda RX4 Wag", value = dplyr::filter(mtcars, uid == "Mazda RX4 Wag"))

## Get "Mazda RX4" specification
repository$get(key = "Mazda RX4")

## Get all the specifications in the repository
repository$get_all_cars()

## Delete "Mazda RX4" specification
repository$del(key = "Mazda RX4")

## Get "Mazda RX4" specification
repository$get(key = "Mazda RX4")
}
\concept{object-relational patterns}
\section{Super class}{
\code{\link[R6P:Singleton]{R6P::Singleton}} -> \code{Repository}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-new}{\code{AbstractRepository$new()}}
\item \href{#method-add}{\code{AbstractRepository$add()}}
\item \href{#method-del}{\code{AbstractRepository$del()}}
\item \href{#method-get}{\code{AbstractRepository$get()}}
}
}
\if{html}{
\out{<details open ><summary>Inherited methods</summary>}
\itemize{
}
\out{</details>}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-new"></a>}}
\if{latex}{\out{\hypertarget{method-new}{}}}
\subsection{Method \code{new()}}{
Instantiate an object
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AbstractRepository$new()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-add"></a>}}
\if{latex}{\out{\hypertarget{method-add}{}}}
\subsection{Method \code{add()}}{
Add an element to the Repository.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AbstractRepository$add(key, value)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{key}}{(\code{character}) Name of the element.}

\item{\code{value}}{(\verb{?}) Value of the element. Note: The values in the
\code{Repository} are not necessarily of the same type. That depends on the
implementation of \code{AbstractRepository}.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-del"></a>}}
\if{latex}{\out{\hypertarget{method-del}{}}}
\subsection{Method \code{del()}}{
Delete an element from the Repository.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AbstractRepository$del(key)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{key}}{(\code{character}) Name of the element.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-get"></a>}}
\if{latex}{\out{\hypertarget{method-get}{}}}
\subsection{Method \code{get()}}{
Retrieve an element from the Repository.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AbstractRepository$get(key)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{key}}{(\code{character}) Name of the element.}
}
\if{html}{\out{</div>}}
}
}
}
