% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PP_4pl.R
\name{PP_4pl}
\alias{PP_4pl}
\title{Estimate Person Parameters for the 4-PL model}
\usage{
PP_4pl(respm, thres, slopes = NULL, lowerA = NULL, upperA = NULL,
  theta_start = NULL, mu = NULL, sigma2 = NULL, type = "wle",
  maxsteps = 40, exac = 0.001, H = 1, ctrl = list())
}
\arguments{
\item{respm}{An integer matrix, which contains the examinees responses. A persons x items matrix is expected.}

\item{thres}{A numeric vector or a numeric matrix which contains the threshold parameter (also known as difficulty parameter or beta parameter) for each item. If a matrix is submitted, the first row must contain only \bold{zeroes}!}

\item{slopes}{A numeric vector, which contains the slope parameters for each item - one parameter per item is expected.}

\item{lowerA}{A numeric vector, which contains the lower asymptote parameters (kind of guessing parameter) for each item.}

\item{upperA}{numeric vector, which contains the upper asymptote parameters for each item.}

\item{theta_start}{A vector which contains a starting value for each person. If NULL is submitted, the starting values are set automatically. If a scalar is submitted, this start value is used for each person.}

\item{mu}{A numeric vector of location parameters for each person in case of MAP or EAP estimation. If nothing is submitted this is set to 0 for each person for MAP estimation.}

\item{sigma2}{A numeric vector of variance parameters for each person in case of MAP or EAP estimation. If nothing is submitted this is set to 1 for each person for MAP estimation.}

\item{type}{Which maximization should be applied? There are five valid entries possible: "mle", "wle", "map", "eap" and "robust". To choose between the methods, or just to get a deeper understanding the papers mentioned below are quite helpful. The default is \code{"wle"} which is a good choice in many cases.}

\item{maxsteps}{The maximum number of steps the NR Algorithm will take. Default = 100.}

\item{exac}{How accurate are the estimates supposed to be? Default is 0.001.}

\item{H}{In case \code{type = "robust"} a Huber ability estimate is performed, and \code{H} modulates how fast the downweighting takes place (for more Details read Schuster & Yuan 2011).}

\item{ctrl}{more controls:
\itemize{
\item \code{killdupli}: Should duplicated response pattern be removed for estimation (estimation is faster)? This is especially resonable in case of a large number of examinees and a small number of items.  Use this option with caution (for map and eap), because persons with different \code{mu} and \code{sigma2} will have different ability estimates despite they responded identically. Default value is \code{FALSE}.

\item \code{skipcheck}: Default = FALSE. If TRUE data matrix and arguments are not checked - this saves time e.g. when you use this function for simulations.

}}
}
\value{
The function returns a list with the estimation results and pretty much everything which has been submitted to fit the model. The estimation results can be found in \code{OBJ$resPP}. The core result is a number_of_persons x 2 matrix, which contains the ability estimate and the SE for each submitted person.
}
\description{
Compute Person Parameters for the 1/2/3/4-PL model and choose between five common estimation techniques: ML, WL, MAP, EAP and a robust estimation. All item parameters are treated as fixed.
}
\details{
With this function you can estimate:
\itemize{
\item \bold{1-PL model} (Rasch model) by submitting: the data matrix, item difficulties and \bold{nothing else}, since the 1-PL model is merely a 4-PL model with: any slope = 1, any lower asymptote = 0 and any upper asymptote = 1!
\item \bold{2-PL model} by submitting: the data matrix, item difficulties and slope parameters. Lower and upper asymptotes are automatically set to 0 und 1 respectively.
\item \bold{3-PL model} by submitting anything except the upper asymptote parameters
\item \bold{4-PL model} ---> submit all parameters ...
}

The probability function of the 4-PL model is:
\deqn{P(x_{ij} = 1 | \hat \alpha_i, \hat\beta_i, \hat\gamma_i, \hat\delta_i, \theta_j ) = \hat\gamma_i + (\hat\delta_i-\hat\gamma_i) \frac{exp(\hat \alpha_i (\theta_{j} - \hat\beta_{i}))}{\,1 + exp(\hat\alpha_i (\theta_{j} - \hat\beta_{i}))}}

In our case \eqn{\theta} is to be estimated, and the four item parameters are assumed as fixed (usually these are estimates of a former scaling procedure).

The 3-PL model is the same, except that \eqn{\delta_i = 1, \forall i}.

In the 2-PL model \eqn{\delta_i = 1, \gamma_i = 0, \forall i}.

In the 1-PL model \eqn{\delta_i = 1, \gamma_i = 0, \alpha_i = 1, \forall i}.

.

The \bold{robust} estimation method, applies a Huber-type estimator (Schuster & Yuan, 2011), which downweights responses to items which provide little information for the ability estimation. First a residuum is estimated and on this basis, the weight for each observation is computed.

residuum:
\deqn{r_i = \alpha_i(\theta - \beta_i)}

weight:

\deqn{w(r_i) =  1 \rightarrow if\, |r_i| \leq H}
\deqn{w(r_i) = H/|r| \rightarrow if\, |r_i| > H}
}
\examples{
################# 4 PL #############################################################

### real data ##########

data(pp_amt)

d <- as.matrix(pp_amt$daten_amt[,-(1:7)])

rd_res <- PP_4pl(respm = d, thres = pp_amt$betas[,2], type = "wle")
summary(rd_res)

rd_res1 <- PP_4pl(respm = d, thres = pp_amt$betas[,2], theta_start = 0,type = "wle")
summary(rd_res1)

### fake data ##########
# smaller ... faster

set.seed(1522)
# intercepts
diffpar <- seq(-3,3,length=12)
# slope parameters
sl     <- round(runif(12,0.5,1.5),2)
la     <- round(runif(12,0,0.25),2)
ua     <- round(runif(12,0.8,1),2)

# response matrix
awm <- matrix(sample(0:1,10*12,replace=TRUE),ncol=12)


## 1PL model ##### 

# MLE
res1plmle <- PP_4pl(respm = awm,thres = diffpar,type = "mle")
# WLE
res1plwle <- PP_4pl(respm = awm,thres = diffpar,type = "wle")
# MAP estimation
res1plmap <- PP_4pl(respm = awm,thres = diffpar,type = "map")
# EAP estimation
res1pleap <- PP_4pl(respm = awm,thres = diffpar,type = "eap")
# robust estimation
res1plrob <- PP_4pl(respm = awm,thres = diffpar,type = "robust")

# summarize results
summary(res1plmle)
summary(res1plwle)
summary(res1plmap)


## 2PL model ##### 

# MLE
res2plmle <- PP_4pl(respm = awm,thres = diffpar, slopes = sl,type = "mle")
# WLE
res2plwle <- PP_4pl(respm = awm,thres = diffpar, slopes = sl,type = "wle")
# MAP estimation
res2plmap <- PP_4pl(respm = awm,thres = diffpar, slopes = sl,type = "map")
# EAP estimation
res2pleap <- PP_4pl(respm = awm,thres = diffpar, slopes = sl,type = "eap")
# robust estimation
res2plrob <- PP_4pl(respm = awm,thres = diffpar, slopes = sl,type = "robust")


## 3PL model ##### 

# MLE
res3plmle <- PP_4pl(respm = awm,thres = diffpar,
                    slopes = sl,lowerA = la,type = "mle")
# WLE
res3plwle <- PP_4pl(respm = awm,thres = diffpar,
                    slopes = sl,lowerA = la,type = "wle")
# MAP estimation
res3plmap <- PP_4pl(respm = awm,thres = diffpar,
                    slopes = sl,lowerA = la,type = "map")
# EAP estimation
res3pleap <- PP_4pl(respm = awm,thres = diffpar,
                    slopes = sl,lowerA = la, type = "eap")


## 4PL model ##### 

# MLE
res4plmle <- PP_4pl(respm = awm,thres = diffpar,
                    slopes = sl,lowerA = la,upperA=ua,type = "mle")
# WLE
res4plwle <- PP_4pl(respm = awm,thres = diffpar,
                    slopes = sl,lowerA = la,upperA=ua,type = "wle")
# MAP estimation
res4plmap <- PP_4pl(respm = awm,thres = diffpar,
                    slopes = sl,lowerA = la,upperA=ua,type = "map")
# EAP estimation
res4pleap <- PP_4pl(respm = awm,thres = diffpar,
                    slopes = sl,lowerA = la,upperA=ua,type = "eap")


## A special on robust estimation:
# it reproduces the example given in Schuster & Ke-Hai 2011:

diffpar <- c(-3,-2,-1,0,1,2,3)

AWM <- matrix(0,7,7)
diag(AWM) <- 1

res1plmle <- PP_4pl(respm = AWM,thres = diffpar, type = "mle")

summary(res1plmle)

res1plrob <- PP_4pl(respm = AWM,thres = diffpar, type = "robust")

summary(res1plrob)
}
\references{
Baker, Frank B., and Kim, Seock-Ho (2004). Item Response Theory - Parameter Estimation Techniques. CRC-Press.

Barton, M. A., & Lord, F. M. (1981). An Upper Asymptote for the Three-Parameter Logistic Item-Response Model.

Birnbaum, A. (1968). Some latent trait models and their use in inferring an examinee's ability. In Lord, F.M. & Novick, M.R. (Eds.), Statistical theories of mental test scores. Reading, MA: Addison-Wesley.

Magis, D. (2013). A note on the item information function of the four-parameter logistic model. Applied Psychological Measurement, 37(4), 304-315.

Samejima, Fumiko (1993). The bias function of the maximum likelihood estimate of ability for the dichotomous response level. Psychometrika,  58, 195-209.

Samejima, Fumiko (1993). An approximation of the bias function of the maximum likelihood estimate of a latent variable for the general case where the item responses are discrete. Psychometrika,  58, 119-138.

Schuster, C., & Yuan, K. H. (2011). Robust estimation of latent ability in item response models. Journal of Educational and Behavioral Statistics, 36(6), 720-735.

Warm, Thomas A. (1989). Weighted Likelihood Estimation Of Ability In Item Response Theory. Psychometrika, 54, 427-450.

Yen, Y.-C., Ho, R.-G., Liao, W.-W., Chen, L.-J., & Kuo, C.-C. (2012). An empirical evaluation of the slip correction in the four parameter logistic models with computerized adaptive testing. Applied Psychological Measurement, 36, 75-87.
}
\seealso{
\link{PPass}, \link{PPall}, \link{PP_gpcm}, \link{JKpp}, \link{PV}
}
\author{
Manuel Reif
}
\keyword{4pl}
\keyword{Parameters,}
\keyword{Person}
