% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kruskalTest.R
\name{kruskalTest}
\alias{kruskalTest}
\alias{kruskalTest.default}
\alias{kruskalTest.formula}
\title{Kruskal-Wallis Rank Sum Test}
\usage{
kruskalTest(x, ...)

\method{kruskalTest}{default}(x, g, dist = c("Chisquare",
  "KruskalWallis", "FDist"), ...)

\method{kruskalTest}{formula}(formula, data, subset, na.action,
  dist = c("Chisquare", "KruskalWallis", "FDist"), ...)
}
\arguments{
\item{x}{a numeric vector of data values, or a list of numeric data
vectors.}

\item{\dots}{further arguments to be passed to or from methods.}

\item{g}{a vector or factor object giving the group for the
corresponding elements of \code{"x"}.
Ignored with a warning if \code{"x"} is a list.}

\item{dist}{the test distribution. Defaults's to \code{"Chisquare"}.}

\item{formula}{a formula of the form \code{response ~ group} where
\code{response} gives the data values and \code{group} a vector or
factor of the corresponding groups.}

\item{data}{an optional matrix or data frame (or similar: see
\code{\link{model.frame}}) containing the variables in the
formula \code{formula}.  By default the variables are taken from
\code{environment(formula)}.}

\item{subset}{an optional vector specifying a 
subset of observations to be used.}

\item{na.action}{a function which indicates what should happen when
the data contain \code{NA}s.  Defaults to \code{getOption("na.action")}.}
}
\value{
A list with class \code{"htest"} containing the following components:
\describe{
 \item{method}{a character string indicating what type of test was performed.}
 \item{data.name}{a character string giving the name(s) of the data.}
 \item{statistic}{the estimated quantile of the test statistic.}
 \item{p.value}{the p-value for the test.}
 \item{parameter}{the parameters of the test statistic, if any.}
 \item{alternative}{a character string describing the alternative hypothesis.}
\item{estimates}{the estimates, if any.}
\item{null.value}{the estimate under the null hypothesis, if any.}
}
}
\description{
Performs a Kruskal-Wallis rank sum test.
}
\details{
For one-factorial designs with non-normally distributed
residuals the Kruskal-Wallis rank sum test can be performed to test
the H\eqn{_0: F_1(x) = F_2(x) = \ldots = F_k(x)} against
the H\eqn{_\mathrm{A}: F_i (x) \ne F_j(x)~ (i \ne j)} with at least
one strict inequality.

As the Kruskal-Wallis H-statistic is assymptotically
chi-squared distributed with \eqn{v = k - 1} degree
of freedom, the default test distribution is consequently
\code{dist = "Chisquare"}. If \code{dist = "KruskalWallis"} is selected,
an incomplete beta approximation is used for the calculation
of p-values as implemented in the function
\code{\link[SuppDists]{pKruskalWallis}} of the package
\pkg{SuppDists}. For \code{dist = "FDist"}
the proposed method of Conover and Imam (1981) is used, which is
equivalent to a one-way ANOVA F-test using rank transformed data
(see examples).
}
\examples{
## Hollander & Wolfe (1973), 116.
## Mucociliary efficiency from the rate of removal of dust in normal
##  subjects, subjects with obstructive airway disease, and subjects
##  with asbestosis.
x <- c(2.9, 3.0, 2.5, 2.6, 3.2) # normal subjects
y <- c(3.8, 2.7, 4.0, 2.4)      # with obstructive airway disease
z <- c(2.8, 3.4, 3.7, 2.2, 2.0) # with asbestosis

datf <- data.frame(
  g = g <- c(rep("ns", length(x)), rep("oad",
      length(y)), rep("a", length(z))),
  x = x <- c(x, y, z))

## Using incomplete beta approximation
kruskalTest(x ~ g, datf, dist="KruskalWallis")

## Using chisquare distribution
kruskalTest(x ~ g, datf, dist="Chisquare")

## Check with kruskal.test from R stats
kruskal.test(x ~ g, datf)

## Using Conover's F
kruskalTest(x ~ g, datf, dist="FDist")

## Check with aov on ranks
anova(aov(rank(x) ~ g, datf))

## Check with oneway.test
oneway.test(rank(x) ~ g, datf, var.equal = TRUE)
}
\references{
Conover, W. J., Iman, R. L. (1981) Rank transformations as a bridge
 between parametric and nonparametric statistics, \emph{The American
   Statistician} \bold{35}, 124--129.

 Sachs, L. (1997) \emph{Angewandte Statistik}. Berlin: Springer.
}
\seealso{
\code{\link{kruskal.test}}, \code{\link[SuppDists]{pKruskalWallis}},
 \code{\link{Chisquare}}, \code{\link{FDist}}
}
