#'Easy volcano plot with Olink theme
#'
#'Generates a volcano plot using the results of the olink_ttest function using ggplot and ggplot2::geom_point.
#'The estimated difference is plotted on the x-axis and the negative 10-log p-value on the y-axis.
#'The horizontal dotted line indicates p-value=0.05.
#'Dots are colored based on the Benjamini-Hochberg adjusted p-value cutoff 0.05 and can optionally be annotated by OlinkID.
#'
#' @param p.val_tbl a data frame of results generated by olink_ttest()
#' @param x_lab Optional. Character value to use as the X-axis label
#' @param olinkid_list Optional. Character vector of proteins (by OlinkID) to label in the plot. If not provided, default is to label all significant proteins.
#' @param ... Optional. Additional arguments for  olink_color_discrete()
#'
#' @return An object of class "ggplot", plotting significance (y-axis) by estimated difference between groups (x-axis) for each protein.
#' @export
#' @examples
#' \donttest{
#'
#' library(dplyr)
#'
#' npx_df <- npx_data1 %>% filter(!grepl('control',SampleID, ignore.case = TRUE))
#' ttest_results <- olink_ttest(df=npx_df,
#'                              variable = 'Treatment',
#'                              alternative = 'two.sided')
#' olink_volcano_plot(ttest_results)}
#' @importFrom magrittr %>%
#' @importFrom dplyr filter pull
#' @importFrom ggplot2 ggplot aes geom_point labs geom_hline
#' @importFrom ggrepel geom_label_repel


olink_volcano_plot <- function (p.val_tbl, x_lab = "Estimate", olinkid_list = NULL, ...)
{

  #checking ellipsis
  if(length(list(...)) > 0){

    ellipsis_variables <- names(list(...))

    if(length(ellipsis_variables) == 1){

      if(!(ellipsis_variables == 'coloroption')){

        stop(paste0('The ... option only takes the coloroption argument. ... currently contains the variable ',
                    ellipsis_variables,
                    '.'))

      }

    }else{

      stop(paste0('The ... option only takes one argument. ... currently contains the variables ',
                  paste(ellipsis_variables, collapse = ', '),
                  '.'))
    }
  }

  if(is.null(olinkid_list)){

    olinkid_list <- p.val_tbl %>%
      dplyr::filter(Threshold == 'Significant') %>%
      dplyr::pull(OlinkID)

  }


  volcano_plot <- p.val_tbl %>%
    ggplot2::ggplot(ggplot2::aes(x = estimate, y = -log10(p.value),
                                 color = Threshold)) +
    ggplot2::geom_point() +
    ggplot2::labs(x = x_lab, y = "-log10(p-value)") +
    ggrepel::geom_label_repel(data = subset(p.val_tbl, OlinkID %in% olinkid_list),
                              ggplot2::aes(label = Assay), box.padding = 1, show.legend = FALSE) +
    ggplot2::geom_hline(yintercept = -log10(0.05), linetype="dotted") +
    OlinkAnalyze::set_plot_theme() +
    OlinkAnalyze::olink_color_discrete(...)


  return(volcano_plot)

}
