#' Run an Experiment Based on the Configuration
#'
#' Executes the experiment and saves the results to an Excel file.
#'
#' @param gptConfig A list containing the configuration for the language model, including the system prompt,
#' model specifications, and token settings.
#' @param savePath The file path where the experiment results will be saved in Excel format.
#' Defaults to './output.xlsx' in the current working directory.
#'
#' @return This function does not return a value but saves the experiment results to the specified Excel file.
#' Upon completion, "Done." will be printed to the console.
#'
#' @export
#'
#' @examples
#' \dontrun{
#'
#' runExperiment(Experiment_config,"./output.xlsx")
#'
#' #The first argument Experiment_config is generated by preCheck() function.
#'
#' Experiment_config <- preCheck(data)
#' }
runExperiment<- function (gptConfig,savePath="./output.xlsx"){
  if(Sys.getenv("model")=="llama"){
    run_llama(gptConfig,savePath)
  }
  else if(Sys.getenv("model")=="whisper-1"){
    if(savePath=="./output.xlsx")
      savePath="./output/"
    run_openai_audio(gptConfig,savePath)
  }
  else{
    run_openai(gptConfig,savePath)
  }
}


run_openai_audio <- function(gptConfig,savePath){

}














#' Internal Execution of the Experiment Scenarios
#'
#' @description
#' This internal function manages the execution of the experiment scenarios based on the gptConfig settings.
#' It iteratively processes the data for each run and trial, interacts with the GPT model,
#' and appends the results in the designated Excel file. It is utilized within the 'runExperiment' function.
#' @importFrom utils setTxtProgressBar txtProgressBar write.csv
#' @import openxlsx
#' @param gptConfig A list containing the configuration for the GPT model, including the system prompt,
#' model specifications, token settings, and experiment mode.
#' @param savePath The file path where the experiment results will be saved in Excel format.
#'
#' @return This function does not return a value but executes the experiment scenarios and
#' compiles the results in an Excel file. It prints "Done." to the console upon completion.
#'
#' @noRd
run_openai<- function(gptConfig,savePath){
  data <- as.data.frame(gptConfig[1])
  systemPrompt <- gptConfig$systemPrompt
  model <- gptConfig$model
  max_tokens <- gptConfig$max_tokens
  temperature <- gptConfig$temperature
  top_p <- gptConfig$top_p
  n <- gptConfig$n
  modality <-gptConfig$modality
  imgDetail <-gptConfig$imgDetail

  csvFlag=FALSE
  if(sub(".*\\.", "", savePath) == 'csv'){
    savePath<-paste0(sub("\\.[^.]*$", "", savePath),".xlsx")
    csvFlag=TRUE
  }





  total_iterations <- nrow(data) * n
  current_progress <- 0
  progress_bar <- utils::txtProgressBar(min = 0, max = total_iterations, style = 3)
  utils::setTxtProgressBar(progress_bar, current_progress)




  data$Response <- NA

  data$N <- NA
  data$Trial <- NA
  data$Message <- NA

  wb <- createWorkbook()
  addWorksheet(wb, "Sheet1")
  writeData(wb, sheet = 1, x = t(colnames(data)), startRow = 1, startCol = 1,colNames = FALSE)
  row_num <- 2

  for (s in unique(data$Session)) {
    s_data <- data[data$Session == s, ]
    # Run loop
    for (r in unique(data$Run)) {
      r_data <- s_data[s_data$Run == r, ]
      messages=list()
      messages=addMessage(messages,"system", systemPrompt)

      message(r)
      # Trial loop
      for (i in seq_len(nrow(r_data))) {

        messages=addMessage(messages,"user",r_data$Prompt[i],modality,imgDetail)
        t_data <- r_data[i,]

        repeat {
          result <- tryCatch({
            content_list = openai_chat(messages, system_prompt = systemPrompt, model, temperature, max_tokens, frequency_penalty = 0.0, presence_penalty = 0.0, top_p, n)
            TRUE

          }, error = function(e) {
            warning(paste("wariming:", e))
            FALSE
          })

          if (!isTRUE(result)) {
            Sys.sleep(6)
          } else {
            break
          }
        }


        if (length(content_list) == 1) {
          content_str <- content_list
        } else {
          content_str <- paste(content_list, collapse = "\n")
        }


        # N responses loop
        for(nr in seq_along(content_list)){{
          t_data$Response <- content_list[nr]
          t_data$N <- nr

          if (n==1){
            messages=addMessage(messages,"assistant",content_list[nr])

          }
          cMessage <- paste(messages, collapse = " ")

          new_row <- c(t_data$Run, t_data$Item, t_data$Condition,t_data$Prompt, t_data$Session,t_data$Response, t_data$N,i,cMessage)
          writeData(wb, sheet = 1, x = t(new_row), startRow = row_num, startCol = 1,colNames = FALSE)
          row_num <- row_num + 1
          current_progress <- current_progress + 1
          setTxtProgressBar(progress_bar, current_progress)

        }
          saveWorkbook(wb, savePath, overwrite = TRUE)
        }
      }
    }
  }

  if(csvFlag){
    csvPath<-paste0(sub("\\.[^.]*$", "", savePath),".csv")

    data_xlsx <- read.xlsx(savePath)

    utils::write.csv(data_xlsx, csvPath, row.names = FALSE)
    fileDeleted <- file.remove(savePath)
  }


  close(progress_bar)
  message("Done.")
}







#' Internal Execution of the Experiment Scenarios
#'
#' @description
#' This internal function manages the execution of the experiment scenarios based on the gptConfig settings.
#' It iteratively processes the data for each run and trial, interacts with the llama model,
#' and appends the results in the designated Excel file. It is utilized within the 'runExperiment' function.
#' @importFrom utils setTxtProgressBar txtProgressBar write.csv
#' @import openxlsx
#' @param gptConfig A list containing the configuration for the llama model, including the system prompt,
#' model specifications, token settings, and experiment mode.
#' @param savePath The file path where the experiment results will be saved in Excel format.
#'
#' @return This function does not return a value but executes the experiment scenarios and
#' compiles the results in an Excel file. It prints "Done." to the console upon completion.
#'
#' @noRd
run_llama<- function(gptConfig,savePath){
  data <- as.data.frame(gptConfig[1])
  systemPrompt <- gptConfig$systemPrompt
  # model <- gptConfig$model
  max_tokens <- gptConfig$max_tokens
  temperature <- gptConfig$temperature
  top_p <- gptConfig$top_p
  n <- 1

  csvFlag=FALSE
  if(sub(".*\\.", "", savePath) == 'csv'){
    savePath<-paste0(sub("\\.[^.]*$", "", savePath),".xlsx")
    csvFlag=TRUE
  }

  total_iterations <- nrow(data) * n
  current_progress <- 0

  progress_bar <- utils::txtProgressBar(min = 0, max = total_iterations, style = 3)

  utils::setTxtProgressBar(progress_bar, current_progress)






  data$Response <- NA

  data$N <- NA
  data$Trial <- NA
  data$Message <- NA

  wb <- createWorkbook()
  addWorksheet(wb, "Sheet1")
  writeData(wb, sheet = 1, x = t(colnames(data)), startRow = 1, startCol = 1,colNames = FALSE)
  row_num <- 2

  for (s in unique(data$Session)) {
    s_data <- data[data$Session == s, ]

    # Run loop
    for (r in unique(data$Run)) {
      r_data <- s_data[s_data$Run == r, ]

   ######################################################
      messages=""
      messages=addMessage(messages,"system", systemPrompt)


   ######################################################

      message(r)
      # Trial loop
      for (i in seq_len(nrow(r_data))) {
        messages=addMessage(messages,"user",r_data$Prompt[i])
        t_data <- r_data[i,]

        repeat {
          result <- tryCatch({
            content_list = llama_chat(messages, temperature, max_tokens,top_p)
            TRUE

          }, error = function(e) {
            warning(paste("wariming:", e))
            FALSE
          })

          if (!isTRUE(result)) {
            Sys.sleep(5)
          } else {
            break
          }
        }

          t_data$Response <- content_list
          t_data$N <- 1


          messages=addMessage(messages,"assistant",content_list)


          cMessage <- paste(messages, collapse = " ")

          new_row <- c(t_data$Run, t_data$Item, t_data$Condition,t_data$Prompt, t_data$Session,t_data$Response, t_data$N,i,cMessage)
          writeData(wb, sheet = 1, x = t(new_row), startRow = row_num, startCol = 1,colNames = FALSE)
          row_num <- row_num + 1
          current_progress <- current_progress + 1
          setTxtProgressBar(progress_bar, current_progress)

          saveWorkbook(wb, savePath, overwrite = TRUE)


      }
    }
  }

  if(csvFlag){
    csvPath<-paste0(sub("\\.[^.]*$", "", savePath),".csv")

    data_xlsx <- read.xlsx(savePath)
    utils::write.csv(data_xlsx, csvPath, row.names = FALSE)

    fileDeleted <- file.remove(savePath)
  }


  close(progress_bar)
  message("Done.")
}




