% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mdmr.R
\name{delta}
\alias{delta}
\title{Compute univariate MDMR effect sizes}
\usage{
delta(X, Y = NULL, dtype = NULL, niter = 10, x.inds = NULL,
  y.inds = NULL, G = NULL, G.list = NULL, ncores = 1, seed = NULL,
  plot.res = F, grayscale = F, cex = 1, y.las = 2)
}
\arguments{
\item{X}{A \eqn{n x p} matrix or data frame of predictors. Factors are
allowed to be included, and will be dummy-coded prior to analysis.}

\item{Y}{Outcome data: \eqn{n x q} matrix of scores along the
dependent variables.}

\item{dtype}{Measure of dissimilarity that will be used by \code{\link{dist}}
to compute the distance matrix based on \code{Y}. As is the case when calling
\code{dist} directly, this must be one of \code{"euclidean"},
\code{"maximum"}, \code{"manhattan"}, \code{"canberra"}, \code{"binary"} or
\code{"minkowski"}, and unambiguous substring can be given.}

\item{niter}{Number of times to permute each outcome item in the procedure
to compute delta. The final result is the average of all \code{niter}
iterations. Higher values of \code{niter} require more computation time, but
result in more precise estimates.}

\item{x.inds}{Vector indicating which columns of X should have their
conditional effect sizes computed. Default value of \code{NULL} results in
all effects being computed, and a value of \code{0} results in no conditional
effects being computed, such that only the omnibus effect sizes will be
reported.}

\item{y.inds}{Vector indicating which columns of Y effect sizes should be
computed on. Default value of \code{NULL} results in all columns being
used.}

\item{G}{Gower's centered similarity matrix computed from \code{D}.
Either \code{D} or \code{G} must be passed to \code{mdmr()}.}

\item{G.list}{List of length \eqn{q} where the \eqn{i^{th}}
element contains the \code{G} matrix computed from distance a matrix that
was computed on a version of \code{Y} where the \eqn{i^{th}}
column has been randomly permuted.}

\item{ncores}{Integer; if \code{ncores} > 1, the \code{\link{parallel}}
package is used to speed computation. Note: Windows users must set
\code{ncores = 1} because the \code{parallel} pacakge relies on forking. See
\code{mc.cores} in the \code{\link{mclapply}} function in the
\code{parallel} pacakge for more details.}

\item{seed}{Integer; sets seed for the permutations of each variable
comprising Y so that results can be replicated.}

\item{plot.res}{Logical; Indicates whether or not a heat-map of the results
should be plotted.}

\item{grayscale}{Logical; Indicates whether or not the heat-map should be
plotted in grayscale.}

\item{cex}{Multiplier for cex.axis, cex.lab, cex.main, and cex that are
passed to the plotted result.}

\item{y.las}{Orientation of labels for the outcome items. Defaults to
vertical (2). Value of 1 prints horizontal labels, and is only recommended
if the multivariate outcome is comprised of few variables.}
}
\value{
A data frame whose rows correspond to the omnibus effects and the
effect of each individual predictor (conditional on the rest), and whose
columns correspond to each outcome variable whose effect sizes are being
quantified. If \code{plot.res = TRUE}, a heat-map is plotted of this data
frame to easily identify the strongest effects. Note that the heatmap is
partitioned into the omnibus effect (first row) and pair-wise effects
(remaining rows), because otherwise the omnibus effect would dominate the
heatmap.
}
\description{
\code{delta} computes permutation-based effect sizes on individual items
comprising the distance matrix outcome used in multivariate distance matrix
regression. It returns the omnibus estimates of delta (i.e. effect size of
the entire design matrix on each outcome) as well as estimates of each
pair-wise effect size (i.e. the effect of each predictor on each outcome
variable, conditional on the rest of the predictors).
}
\details{
See McArtor & Lubke (submitted) for a detailed description of how delta is
computed. Note that it is a relative measure of effect, quantifying which
effects are strong (high values of delta) and weak (low values of delta)
within a single analysis, but estimates of delta cannot be directly compared
across different datasets.

There are two options for using this function. The first option is to
specify the predictor matrix \code{X}, the outcome matrix \code{Y}, the
distance type \code{dtype} (supported by "dist" in R), and number of
iterations \code{niter}. This option conducts the permutation of each Y-item
\code{niter} times (to average out random association in each permutation)
and reports the median estimates of delta over the \code{niter} reps.

The second option is to specify \code{X}, \code{G}, and \code{G.list}, a
list of G matrices where the permutation has already been done for each item
comprising Y. The names of the elements in \code{G.list} should correspond
to the names of the variables that were permuted. This option is implemented
so that delta can be computed when MDMR is being used in conjunction with
distance metrics not supported by \code{dist}.
}
\examples{
data(mdmrdata)
# --- Method 1 --- #
delta(X.mdmr, Y = Y.mdmr, dtype = 'euclidean', niter = 1, seed = 12345)

# --- Method 2 --- #
D <- dist(Y.mdmr, method = 'euclidean')
G <- gower(D)
q <- ncol(Y.mdmr)
G.list <- vector(mode = 'list', length = q)
names(G.list) <- names(Y.mdmr)
for(i in 1:q){
   Y.shuf <- Y.mdmr
   Y.shuf[,i] <- sample(Y.shuf[,i])
   G.list[[i]] <- gower(dist(Y.shuf, method = 'euclidean'))
}
delta(X.mdmr, G = G, G.list = G.list)

}
\author{
Daniel B. McArtor (dmcartor@nd.edu) [aut, cre]
}
\references{
McArtor, D.B. & Lubke, G.H. (submitted). Extending
multivariate distance matrix regression with an effect size measure and the
distribution of the test statistic.
}

