% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/run.functions.R
\name{mbnma.run}
\alias{mbnma.run}
\title{Run MBNMA dose-response models}
\usage{
mbnma.run(network, fun = "linear", beta.1 = "rel", beta.2 = NULL,
  beta.3 = NULL, method = "common", class.effect = list(),
  cor = TRUE, var.scale = NULL, user.fun = NULL,
  parameters.to.save = NULL, pd = "pv", parallel = TRUE,
  likelihood = NULL, link = NULL, priors = NULL, model.file = NULL,
  n.iter = 10000, n.chains = 3, n.burnin = floor(n.iter/2),
  n.thin = max(1, floor((n.iter - n.burnin)/1000)), arg.params = NULL,
  ...)
}
\arguments{
\item{network}{An object of class \code{mbnma.network}.}

\item{fun}{A string specifying a functional form to be assigned to the
dose-response. Options are given in \code{details}.}

\item{beta.1}{Refers to dose-parameter(s) specified within the dose-response function.
Can take either \code{"rel"}, \code{"common"}, \code{"random"}, or be assigned a numeric value (see details).}

\item{beta.2}{Refers to dose-parameter(s) specified within the dose-response function.
Can take either \code{"rel"}, \code{"common"}, \code{"random"}, or be assigned a numeric value (see details).}

\item{beta.3}{Refers to dose-parameter(s) specified within the dose-response function.
Can take either \code{"rel"}, \code{"common"}, \code{"random"}, or be assigned a numeric value (see details).}

\item{method}{Can take either \code{"common"} or \code{"random"} to indicate whether relative effects
should be modelled with between-study heterogeneity or not (see details).}

\item{class.effect}{A list of named strings that determines which dose-response
parameters to model with a class effect and what that effect should be
(\code{"common"} or \code{"random"}). Element names should match dose-response parameter names (which will therefore
depend on whether or not a wrapper function has been used for \code{mbnma.run()}).
For example: \code{list("beta.2"="fixed", "beta.3"="random")} when using
\code{mbnma.run()} or \code{list("ed50"="fixed", "hill"="random")} when using \code{mbnma.emax.hill()}.}

\item{cor}{A boolean object that indicates whether correlation should be modelled
between relative effect dose-response parameters (\code{TRUE}) or not (\code{FALSE}). This is
automatically set to \code{FALSE} if class effects are modelled.}

\item{var.scale}{A numeric vector indicating the relative scale of variances between
correlated dose-response parameters when relative effects are modelled on more than
one dose-response parameter and \code{cor=TRUE} (see details). Each element of
the vector refers to the relative scale of each of the dose-response parameters that is
modelled using relative effects.}

\item{user.fun}{A string specifying any relationship including \code{dose} and
one/several of: \code{beta.1}, \code{beta.2}, \code{beta.3}.}

\item{parameters.to.save}{A character vector containing names of parameters
to monitor in JAGS}

\item{pd}{Can take either:
\itemize{
\item \code{pv} only pV will be reported (as automatically outputted by \code{R2jags}).
\item \code{plugin} calculates pD by the plug-in
method \insertCite{spiegelhalter2002}{MBNMAdose}. It is faster, but may output negative
non-sensical values, due to skewed deviances that can arise with non-linear models.
\item \code{pd.kl} calculates pD by the Kullback-Leibler divergence \insertCite{plummer2008}{MBNMAdose}. This
will require running the model for additional iterations but
will always produce a positive result.
\item \code{popt} calculates pD using an optimism adjustment which allows for calculation
of the penalized expected deviance \insertCite{plummer2008}{MBNMAdose}
}}

\item{parallel}{A boolean value that indicates whether JAGS should be run in
parallel (\code{TRUE}) or not (\code{FALSE}). If \code{TRUE} then the number of cores to
use is automatically calculated.}

\item{likelihood}{A string indicating the likelihood to use in the model. Can take either \code{"binomial"},
\code{"normal"} or \code{"poisson"}. If left as \code{NULL} the likelihood will be inferred from the data.}

\item{link}{A string indicating the link function to use in the model. Can take any link function
defined within JAGS (e.g. \code{"logit"}, \code{"log"}, \code{"probit"}, \code{"cloglog"}) or be assigned the value \code{"identity"} for
and identity link function. If left as \code{NULL} the link function will be automatically assigned based
on the likelihood.}

\item{priors}{A named list of parameter values (without indices) and
replacement prior distribution values given as strings
\strong{using distributions as specified in JAGS syntax} (see examples).}

\item{model.file}{A JAGS model written as a character object that can be used
to overwrite the JAGS model that is automatically written based on the
specified options.}

\item{n.iter}{number of total iterations per chain (including burn in; default: 15000)}

\item{n.chains}{number of Markov chains (default: 3)}

\item{n.burnin}{length of burn in, i.e. number of iterations to discard at the
beginning. Default is `n.iter/2``, that is, discarding the first half of the
simulations. If n.burnin is 0, jags() will run 100 iterations for adaption.}

\item{n.thin}{thinning rate. Must be a positive integer. Set \code{n.thin > 1`` to save memory and computation time if n.iter is large. Default is }max(1, floor(n.chains * (n.iter-n.burnin) / 1000))`` which will only thin if there are at least 2000
simulations.}

\item{arg.params}{Contains a list of arguments sent to \code{mbnma.run()} by dose-response
specific wrapper functions}

\item{...}{Arguments to be sent to R2jags.}
}
\value{
An object of S3 \code{class(c("mbnma", "rjags"))} containing parameter
results from the model. Can be summarized by \code{print()} and can check
traceplots using \code{R2jags::traceplot()} or various functions from the package \code{mcmcplots}.

Nodes that are automatically monitored (if present in the model) have the
following interpretation. These will have an additional suffix that relates
to the name/number of the dose-response parameter to which they correspond
(e.g. \code{d.ed50} or \code{d.1}):
\itemize{
\item \code{d} The pooled effect for each agent for a given dose-response
parameter. Will be estimated by the model if dose-response parameters (\code{beta.1},
\code{beta.2}, \code{beta.3}) are set to \code{"rel"}.
\item \code{sd} (without a suffix) - the between-study SD (heterogeneity) for relative effects, reported if
\code{method="random"}.
\item \code{D} The class effect for each class for a given dose-response
parameter. Will be estimated by the model if specified in \code{class.effect}.
\item \code{sd.D} The within-class SD for different agents within the same class. Will
be estimated by the model if any dose-response parameter in \code{class.effect} is
set to \code{"random"}.
\item \code{beta} The absolute value of a given dose-response parameter across the whole
network (does not vary by agent/class). Will be estimated by the model if
dose-response parameters (\code{beta.1}, \code{beta.2}, \code{beta.3}) are set to \code{"common"}
or \code{"random"}.
\item \code{sd} (with a suffix) - the between-study SD (heterogeneity) for absolute dose-response
parameters, reported if \code{beta.1}, \code{beta.2} or \code{beta.3} are set to \code{"random"}
\item \code{totresdev} The residual deviance of the model
\item \code{deviance} The deviance of the model
}

If there are errors in the JAGS model code then the object will be a list
consisting of two elements - an error message from JAGS that can help with
debugging and \code{model.arg}, a list of arguments provided to \code{mbnma.run()}
which includes \code{jagscode}, the JAGS code for the model that can help
users identify the source of the error.
}
\description{
Fits a Bayesian dose-response for model-based network meta-analysis
(MBNMA) that can account for multiple doses of different agents by
applying a desired dose-response function. Follows the methods
of \insertCite{mawdsley2016;textual}{MBNMAdose}.
}
\details{
When relative effects are modelled on more than one dose-response parameter and
\code{cor = TRUE}, correlation between the dose-response parameters is automatically
estimated using a vague Wishart prior. This prior can be made slightly more informative
by specifying the relative scale of variances between the dose-response parameters using
\code{var.scale}.
}
\section{Dose-response parameters}{

\itemize{
\item \code{"rel"} implies that relative effects should be pooled for this dose-response
parameter, that vary by agent.
\item \code{"common"} implies that all studies estimate the same true absolute effect
(akin to a "fixed effects" meta-analysis) across the whole network
\item \code{"random"} implies that all studies estimate a separate true absolute effect, but
that each of these true effects vary randomly around a true mean effect. This
approach allows for modelling of between-study heterogeneity.
\item \code{numeric()} Assigned a numeric value. It indicates that
this dose-response parameter should not be estimated from the data but should be
assigned the numeric value determined by the user. This can be useful for fixing
specific dose-response parameters (e.g. Hill parameters in Emax functions) to a value.
}
}

\section{Dose-response function}{

Several general dose-response functions are provided, but a
user-defined dose-response relationship can instead be used.

Built-in time-course functions are:
\itemize{
\item \code{"linear"}: \code{beta.1} refers to the gradient
\item \code{"exponential"}: \code{beta.1} refers to the rate of gain/decay
\item \code{"emax"} (emax without a Hill parameter): \code{beta.1} refers to
Emax parameter, \code{beta.2} refers to ET50 parameter
\item \code{"emax.hill"} (emax with a Hill parameter): \code{beta.1} refers to Emax parameter, \code{beta.2} refers
to ET50 parameter, \code{beta.3} refers to Hill parameter
\item \code{"nonparam.up"} (monotonically increasing non-parametric dose-response relationship following
the method of \insertCite{owen2015;textual}{MBNMAdose})
\item \code{"nonparam.down"} (monotonically decreasing non-parametric dose-response relationship following
the method of \insertCite{owen2015;textual}{MBNMAdose})
\item \code{"user"} (user-defined function: \code{user.fun} must be specified in arguments)
}
}

\examples{
\donttest{
# Using the triptans data
network <- mbnma.network(HF2PPITT)


######## Dose-response functions ########

# Fit a dose-response MBNMA with a linear function and common treatment effects
result <- mbnma.run(network, fun="linear", beta.1="rel", method="common")

# Fit a dose-response MBNMA with an exponential function and random treatment effects
result <- mbnma.run(network, fun="exponential", beta.1="rel", method="random")

# Fit a user-defined function (quadratic)
fun.def <- "(beta.1 * dose) + (beta.2 * (dose^2))"
result <- mbnma.run(network, fun="user", user.fun=fun.def,
              beta.1="rel", beta.2="rel", method="common")

# Fit an Emax function with a single random (exchangeable) parameter estimated
#for ED50 and common treatment effects on relative Emax effects
result <- mbnma.run(network, fun="emax",
              beta.1="rel", beta.2="random", method="common")

# Fit an Emax function with a Hill parameter, with a fixed value for the Hill parameter
#provided to the model and random relative effects on Emax and ED50 (which will
#therefore be modelled with a correlation between them).
result <- mbnma.run(network, fun="emax.hill",
              beta.1="rel", beta.2="rel", beta.3=5, method="random")


########## Class effects ##########

# Generate a dataset with one class for active treatments and one for placebo
class.df <- HF2PPITT
class.df$class <- ifelse(class.df$agent=="placebo", "placebo", "active")
netclass <- mbnma.network(class.df)

# Fit an Emax function with common relative effects on Emax and ED50 and
#a random class effect on ED50.
result <- mbnma.run(netclass, fun="emax",
              beta.1="rel", beta.2="rel", method="common",
              class.effect=list(beta.2="random"))


####### Priors #######

# Obtain priors from an Emax function with random relative effects on Emax and ED50
result <- mbnma.run(network, fun="emax",
              beta.1="rel", beta.2="rel", method="random")
print(result$model.arg$priors)

# Set new more informative prior distributions
newpriors <- list(sd = "dnorm(0,0.5) T(0,)",
                 inv.R = "dwish(Omega[,],100)")

result <- mbnma.run(network, fun="emax",
              beta.1="rel", beta.2="rel", method="random",
              priors=newpriors)


########## Sampler options ##########

# Change the number of MCMC iterations, the number of chains, and the thin
result <- mbnma.run(network, fun="exponential", beta.1="rel", method="random",
              n.iter=5000, n.thin=5, n.chains=4)

# Calculate effective number of parameters via plugin method
result <- mbnma.run(network, fun="exponential", beta.1="rel", method="random",
              pd="plugin")

# Calculate effective number of parameters via Kullback-Leibler method
result <- mbnma.run(network, fun="exponential", beta.1="rel", method="random",
              pd="pd.kl")


####### Examine MCMC diagnostics (using mcmcplots package) #######

# Density plots
mcmcplots::denplot(result)

# Traceplots
mcmcplots::traplot(result)

# Caterpillar plots
mcmcplots::caterplot(result, "d.1")


########## Output ###########

# Print R2jags output and summary
print(result)
summary(result)

# Plot forest plot of results
plot(result)
}

}
\references{
\insertAllCited
}
