% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MOLindley.R
\name{MOLindley}
\alias{MOLindley}
\alias{dmolindley}
\alias{hmolindley}
\alias{pmolindley}
\alias{qmolindley}
\alias{rmolindley}
\title{Marshall-Olkin Extended Lindley Distribution}
\source{
[dpqh]molindley are calculated directly from the definitions. \code{rmolindley} uses the inverse transform method.
}
\usage{
dmolindley(x, theta, alpha, log = FALSE)

pmolindley(q, theta, alpha, lower.tail = TRUE, log.p = FALSE)

qmolindley(p, theta, alpha, lower.tail = TRUE, log.p = FALSE)

rmolindley(n, theta, alpha)

hmolindley(x, theta, alpha, log = FALSE)
}
\arguments{
\item{x, q}{vector of positive quantiles.}

\item{theta, alpha}{positive parameters.}

\item{log, log.p}{logical. If TRUE, probabilities p are given as log(p).}

\item{lower.tail}{logical. If TRUE (default) \eqn{P(X \leq x)} are returned, otherwise \eqn{P(X > x)}.}

\item{p}{vector of probabilities.}

\item{n}{number of observations. If \code{length(n) > 1}, the length is taken to be the number required.}
}
\value{
\code{dmolindley} gives the density, \code{pmolindley} gives the distribution function, \code{qmolindley} gives the quantile function, \code{rmolindley} generates random deviates and \code{hmolindley} gives the hazard rate function.

Invalid arguments will return an error message.
}
\description{
Density function, distribution function, quantile function, random numbers generation and hazard rate function for the Marshall-Olkin extended Lindley distribution with parameters theta and alpha.
}
\details{
Probability density function
\deqn{f\left( x\mid \theta,\alpha \right) =\frac{\alpha \theta^{2}(1+x)e^{-\theta x}}{(1+\theta )\left[ 1-\overline{\alpha }\left( 1+\frac{\theta x}{1+\theta }\right) e^{-\theta x}\right] ^{2}}}

Cumulative distribution function
\deqn{F\left( x\mid \theta,\alpha \right) =1-\frac{\alpha \left( 1+\frac{\theta x}{1+\theta }\right) e^{-\theta x}}{1-\overline{\alpha }\left( 1+\frac{\theta x}{1+\theta }\right) e^{-\theta x}}}

Quantile function
\deqn{Q(p\mid \theta,\alpha )=-1-\frac{1}{\theta }-\frac{1}{\theta }W_{-1}\left( \frac{(\theta +1)}{e^{1 + \theta}}\frac{(p-1)}{\left( 1-\overline{\alpha }p\right) }\right)}

Hazard rate function
\deqn{h\left( x\mid \theta,\alpha \right) =\frac{\theta ^{2}\left( 1+x\right) }{\left( 1+\theta +\theta x\right) \left[ 1-\overline{\alpha }\left( 1+\frac{\theta x}{1+\theta }\right) e^{-\theta x}\right] }}

where \eqn{W_{-1}} denotes the negative branch of the Lambert W function and \eqn{\overline{\alpha}=(1 - \alpha)}.

\bold{Particular case:} \eqn{\alpha=1} the one-parameter Lindley distribution.
}
\examples{
set.seed(1)
x <- rmolindley(n = 1000, theta = 5, alpha = 5)
R <- range(x)
S <- seq(from = R[1], to = R[2], by = 0.1)
plot(S, dmolindley(S, theta = 5, alpha = 5), xlab = 'x', ylab = 'pdf')
hist(x, prob = TRUE, main = '', add = TRUE)

p <- seq(from = 0.1, to = 0.9, by = 0.1)
q <- quantile(x, prob = p)
pmolindley(q, theta = 5, alpha = 5, lower.tail = TRUE)
pmolindley(q, theta = 5, alpha = 5, lower.tail = FALSE)
qmolindley(p, theta = 5, alpha = 5, lower.tail = TRUE)
qmolindley(p, theta = 5, alpha = 5, lower.tail = FALSE)

library(fitdistrplus)
fit <- fitdist(x, 'molindley', start = list(theta = 5, alpha = 5))
plot(fit)


}
\author{
Josmar Mazucheli \email{jmazucheli@gmail.com}

Larissa B. Fernandes \email{lbf.estatistica@gmail.com}
}
\references{
do Espirito Santo, A. P. J., Mazucheli, J., (2015). Comparison of estimation methods for the Marshall-Olkin extended Lindley distribution. \emph{Journal of Statistical Computation and Simulation}, \bold{85}, (17), 3437-3450.

Ghitany, M. E., Al-Mutairi, D. K., Al-Awadhi, F. A. and Al-Burais, M. M., (2012). Marshall-Olkin extended Lindley distribution and its application. \emph{International Journal of Applied Mathematics}, \bold{25}, (5), 709-721.

Marshall, A. W., Olkin, I. (1997). A new method for adding a parameter to a family of distributions with application to the exponential and Weibull families. \emph{Biometrika}, \bold{84}, 3, 641.652.
}
\seealso{
\code{\link[LambertW]{W}}, \code{\link[LindleyR]{Lindley}}.
}

