% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulateSSM.R
\name{simulateSSM}
\alias{simulateSSM}
\title{Simulation of a Gaussian State Space Model}
\usage{
simulateSSM(object, type = c("states", "signals", "disturbances",
  "observations", "epsilon", "eta"), filtered = FALSE, nsim = 1,
  antithetics = FALSE, conditional = TRUE)
}
\arguments{
\item{object}{Gaussian state space object of class \code{SSModel}.}

\item{type}{What to simulate.}

\item{filtered}{Simulate from \eqn{p(\alpha_t|y_{t-1},...,y_1)}{p(\alpha[t]|y[t-1],...,y[1])}
instead of \eqn{p(\alpha|y)}.}

\item{nsim}{Number of independent samples. Default is 1.}

\item{antithetics}{Use antithetic variables in simulation. Default is \code{FALSE}.}

\item{conditional}{Simulations are conditional to data. If \code{FALSE}, the
states having exact diffuse initial distribution (as defined by \code{P1inf}
are fixed to corresponding values of \code{a1}. See details.}
}
\value{
An n x k x nsim array containing the simulated series, where k is number of observations,
  signals, states or disturbances.
}
\description{
Function \code{simulateSMM} simulates states, signals, disturbances or missing observations of
the Gaussian state space model either conditional on the data (simulation smoother) or
unconditionally.
}
\details{
Simulation smoother algorithm is based on article by J. Durbin and S.J. Koopman (2002).
The simulation filter (\code{filtered = TRUE}) is a straightforward modification
of the simulations smoother, where only filtering steps are performed.

Function can use two antithetic variables, one for location and other for scale, so output
contains four blocks of simulated values which correlate which each other (ith block correlates
negatively with (i+1)th block, and positively with (i+2)th block etc.).

Note that KFAS versions 1.2.0 and older, for unconditional simulation the initial
distribution of states was fixed so that \code{a1} was set to the smoothed estimates
of the first state and the initial variance was set to zero. Now original
\code{a1} and \code{P1} are used, and \code{P1inf} is ignored (i.e. diffuse states are
fixed to corresponding elements of \code{a1}).
}
\examples{

model <- SSModel(matrix(NA, 100, 1) ~ SSMtrend(1, 1, P1inf = 0), H = 1)

set.seed(123)
sim <- simulateSSM(model, "obs", nsim = 2, antithetics = TRUE)
# first time points
sim[1,,]
# correlation structure between simulations with two antithetics
cor(sim[,1,])

out_NA <- KFS(model, filtering = "none", smoothing = "state")
model["y"] <- sim[, 1, 1]
out_obs <- KFS(model, filtering = "none", smoothing = "state")

set.seed(40216)
# simulate states from the p(alpha | y)
sim_conditional <- simulateSSM(model, nsim = 10, antithetics = TRUE)

# mean of the simulated states is exactly correct due to antithetic variables
mean(sim_conditional[2, 1, ])
out_obs$alpha[2]
# for variances more simulations are needed
var(sim_conditional[2, 1, ])
out_obs$V[2]

set.seed(40216)
# no data, simulations from p(alpha)
sim_unconditional <- simulateSSM(model, nsim = 10, antithetics = TRUE,
  conditional = FALSE)
mean(sim_unconditional[2, 1, ])
out_NA$alpha[2]
var(sim_unconditional[2, 1, ])
out_NA$V[2]

ts.plot(cbind(sim_conditional[,1,1:5], sim_unconditional[,1,1:5]),
  col = rep(c(2,4), each = 5))
lines(out_obs$alpha, lwd=2)

}
\references{
Durbin J. and Koopman, S.J. (2002). A simple and efficient simulation smoother for
  state space time series analysis, Biometrika, Volume 89, Issue 3
}

