% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/imagefusion_task.R
\name{imagefusion_task}
\alias{imagefusion_task}
\title{Perform time-series image fusion}
\usage{
imagefusion_task(
  ...,
  filenames_high,
  filenames_low,
  dates_high,
  dates_low,
  dates_pred,
  filenames_pred = NULL,
  singlepair_mode = "ignore",
  method = "starfm",
  high_date_prediction_mode = "ignore",
  verbose = FALSE,
  output_overview = FALSE,
  out_dir = NULL
)
}
\arguments{
\item{...}{Further arguments specific to the chosen \code{method}. See the documentation of the methods for a detailed description.}

\item{filenames_high}{A character vector of the filenames of the high resolution input images.}

\item{filenames_low}{A character vector of the filenames of the low resolution input images.}

\item{dates_high}{Numeric. An integer vector of the dates associated with the \code{filenames_high}. Must match \code{filenames_high} in length and order.}

\item{dates_low}{Numeric. An integer vector of the dates associated with the \code{filenames_low}. Must match \code{filenames_low} in length and order.}

\item{dates_pred}{Numeric. An integer vector dates for which an output should be generated.}

\item{filenames_pred}{(Optional) A character vector of the filenames for the predicted images. If unspecified, filenames will be created from \code{out_dir} and \code{pred_dates}.}

\item{singlepair_mode}{(Optional) How should singlepair predictions (those \code{dates_pred}, which do not lie between two dates with a high&low pair) be handled? \itemize{
\item{ignore: No prediction will be performed for those dates. This is the default.}
\item{mixed: Use doublepair mode where possible, and singlepair mode otherwise (only supported for \code{method} fitfc and starfm)}
\item{all: Predict all dates in singlepair mode, using the closest pair date if between pairs (only supported for \code{method} fitfc and starfm)}}}

\item{method}{(Optional) The algorithm which is used for the fusion. \itemize{
\item{starfm: STARFM stands for spatial and temporal adaptive reflectance fusion model. It requires a relatively low amount of computation time for prediction. Supports singlepair and doublepair modes. See \link[ImageFusion]{starfm_job}.}
\item{estarfm: ESTARFM stands for enhanced spatial and temporal adaptive reflectance fusion model so it claims to be an enhanced STARFM. It can yield better results in some situations. Only supports doublepair mode. See \link[ImageFusion]{estarfm_job}.}
\item{fitfc: Fit-FC is a three-step method consisting of regression model fitting (RM fitting), spatial filtering (SF) and residual compensation (RC). It requires a relatively low amount of computation time for prediction. Supports singlepair or a pseudo-doublepair mode(For predictions between two pair dates, predictions will be done twice, once for each of the pair dates). See \link[ImageFusion]{fitfc_job}. This is the default algorithm.}
}}

\item{high_date_prediction_mode}{(Optional) How to proceed for predictions on those dates which have high resolution images available? \itemize{
\item{ignore: Output nothing for those dates. This is the default.}
\item{copy: Directly copy the high resolution input images to the output path.}
\item{force: Use the chosen algorithm to predicts the high resolution input images on themselves. This takes additional computation time, but ensures that the outputs are consistent with the genuinely predicted outputs.}
}}

\item{verbose}{(Optional) Output additional intermediate progress reports? Default is "false".}

\item{output_overview}{(Optional) Should a summary of the task be printed to console, and a \link[ggplot2]{ggplot} overview be returned? Default is "false".}

\item{out_dir}{(Optional) A directory in which the predicted images will be saved. Will be created if it does not exist. By default, creates a directory "Outputs" in the R temp directory (see \link{tempdir}).}
}
\value{
A ggplot overview of the tasks (If \code{output_overview} is "true")
}
\description{
The main function of the ImageFusion Package, intended for the fusion of images based on a time-series of inputs.
}
\details{
The function firstly seeks among the inputs for pair dates, which are dates for which have both a high resolution image and a low resolution image are available. It then splits the task into a number of self-contained \emph{jobs} which handle the fusion between these pair dates using the paired images as anchors. These jobs can also be called directly via their respective functions, see \code{method}.
}
\examples{
# Load required libraries
library(ImageFusion)
library(raster)
# Get filesnames of high resolution images
landsat <- list.files(
  system.file("landsat/filled",
              package = "ImageFusion"),
  ".tif",
  recursive = TRUE,
  full.names = TRUE
)

# Get filesnames of low resolution images
modis <- list.files(
  system.file("modis",
              package = "ImageFusion"),
  ".tif",
  recursive = TRUE,
  full.names = TRUE
)
# Create output directory in temporary folder
out_dir <- file.path(tempdir(),"Outputs")
if(!dir.exists(out_dir)) dir.create(out_dir, recursive = TRUE)
# Run the fusion on the entire time series
imagefusion_task(filenames_high = landsat,
                 dates_high = c(68,77,93,100),
                 filenames_low = modis,
                 dates_low = 68:93,
                 dates_pred = c(65,85,95),
                 out_dir = out_dir)
# remove the output directory
unlink(out_dir,recursive = TRUE)


}
\references{
Gao, Feng, et al. "On the blending of the Landsat and MODIS surface reflectance: Predicting daily Landsat surface reflectance." IEEE Transactions on Geoscience and Remote sensing 44.8 (2006): 2207-2218.

Wang, Qunming, and Peter M. Atkinson. "Spatio-temporal fusion for daily Sentinel-2 images." Remote Sensing of Environment 204 (2018): 31-42.

Zhu, X., Chen, J., Gao, F., Chen, X., & Masek, J. G. (2010). An enhanced spatial and temporal adaptive reflectance fusion model for complex heterogeneous regions. Remote Sensing of Environment, 114(11), 2610-2623.
}
\author{
Christof Kaufmann (C++), Johannes Mast (R)
}
