% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SuppressLinkedTables.R
\name{SuppressLinkedTables}
\alias{SuppressLinkedTables}
\title{Consistent Suppression of Linked Tables}
\usage{
SuppressLinkedTables(
  data = NULL,
  fun,
  ...,
  withinArg = NULL,
  linkedGauss = "consistent",
  recordAware = TRUE,
  iterBackTracking = Inf,
  whenEmptyUnsuppressed = NULL,
  lpPackage = NULL
)
}
\arguments{
\item{data}{The \code{data} argument to \code{fun}. When NULL \code{data} must be included in  \code{withinArg}.}

\item{fun}{A function: \code{\link{GaussSuppressionFromData}} or one of its wrappers such as
\code{\link{SuppressSmallCounts}} and \code{\link{SuppressDominantCells}}.}

\item{...}{Arguments to \code{fun} that are kept constant.}

\item{withinArg}{A list of named lists. Arguments to \code{fun} that are not kept constant.
If \code{withinArg} is named, the names will be used as names in the output list.}

\item{linkedGauss}{Specifies the strategy for protecting linked tables. Possible values are:
\itemize{
\item \code{"consistent"} (default): All linked tables are protected by a single call to \code{GaussSuppression()}.
The algorithm internally constructs a block diagonal model matrix and handles common cells
consistently across tables.
\item \code{"local"}: Each table is protected independently by a separate call to \code{GaussSuppression()}.
\item \code{"back-tracking"}: Iterative approach where each table is protected via \code{GaussSuppression()},
and primary suppressions are adjusted based on secondary suppressions from other tables across
iterations.
\item \code{"local-bdiag"}: Produces the same result as \code{"local"}, but uses a single call to
\code{GaussSuppression()} with a block diagonal matrix. It does not apply the linked-table methodology.
}}

\item{recordAware}{If \code{TRUE} (default), the suppression procedure will ensure consistency
across cells that aggregate the same underlying records,
even when their variable combinations differ.
When \code{TRUE}, \code{data} cannot be included in  \code{withinArg}.}

\item{iterBackTracking}{Maximum number of back-tracking iterations.}

\item{whenEmptyUnsuppressed}{Parameter to \code{\link[SSBtools]{GaussSuppression}}.
This is about a helpful message
\emph{"Cells with empty input will never be secondary suppressed. Extend input data with zeros?"}
Here, the default is set to \code{NULL} (no message), since preprocessing of the model matrix
may invalidate the assumptions behind this message.}

\item{lpPackage}{Currently ignored. If specified, a warning will be issued.}
}
\value{
A list of data frames, or, if \code{withinArg} is \code{NULL}, the ordinary output from \code{fun}.
}
\description{
Provides alternatives to global protection for linked tables through
methods that may reduce the computational burden.
}
\details{
The reason for introducing the new method \code{"consistent"}, which has not yet been extensively tested in practice,
is to provide something that works better than \code{"back-tracking"}, while still offering equally strong protection.

Note that for singleton methods of the \emph{elimination} type (see \code{\link[SSBtools:NumSingleton]{SSBtools::NumSingleton()}}), \code{"back-tracking"} may lead to
the creation of a large number of redundant secondary cells. This is because, during the method's iterations,
all secondary cells are eventually treated as primary. As a result, protection is applied to prevent a singleton
contributor from inferring a secondary cell that was only included to protect that same contributor.

Note that the frequency singleton methods \code{"subSpace"}, \code{"anySum0"}, and \code{"anySumNOTprimary"} are currently not implemented
and will result in an error.
As a result, the \code{singletonZeros} parameter in the \code{SuppressDominantCells()} function cannot be set to \code{TRUE},
and the \code{SuppressKDisclosure()} function is not available for use.
Also note that automatic forcing of \code{"anySumNOTprimary"} is disabled.
That is, \code{\link[SSBtools:GaussSuppression]{SSBtools::GaussSuppression()}} is called with \code{auto_anySumNOTprimary = FALSE}.
See the parameter documentation for an explanation of why \code{FALSE} is required.

The combination of intervals with the various linked table strategies is not yet implemented,
so the \code{lpPackage} parameter is currently ignored.
}
\note{
Note on differences between \code{SuppressLinkedTables()} and alternative approaches.
By \emph{alternatives}, we refer to using the \code{linkedGauss} parameter via \code{GaussSuppressionFromData()}, its wrappers, or through \code{tables_by_formulas()}, as shown in the examples below.
\itemize{
\item Alternatives can be used when only the \code{formula} parameter varies between the linked tables.
\item \code{SuppressLinkedTables()} creates several smaller model matrices, which may be combined into a single block-diagonal matrix. A large overall matrix is never created.
\item With the alternatives, a large overall matrix is created first. Smaller matrices are then derived from it. If the size of the full matrix is a bottleneck, \code{SuppressLinkedTables()} is the better choice.
\item The \code{"global"} method is available with the alternatives, but not with \code{SuppressLinkedTables()}.
\item Due to differences in candidate ordering, the two methods may not always produce identical results. With the alternatives, candidate order is constructed globally across all cells (as with the global method).
In contrast, \code{SuppressLinkedTables()} uses a locally determined candidate order within each table.  The ordering across tables
is coordinated to ensure the method works, but it is not based on a strictly defined global order.
This may lead to some differences.
}
}
\examples{

### The first example can be performed in three ways
### Alternatives are possible since only the formula parameter varies between the linked tables
 
a <- SuppressLinkedTables(data = SSBtoolsData("magnitude1"), # With trick "sector4 - sector4" and 
                 fun = SuppressDominantCells,        # "geo - geo" to ensure same names in output
                 withinArg = list(list(formula = ~(geo + eu) * sector2 + sector4 - sector4), 
                                  list(formula = ~eu:sector4 - 1 + geo - geo), 
                                  list(formula = ~geo + eu + sector4 - 1)), 
                 dominanceVar  = "value", 
                 pPercent = 10, 
                 contributorVar = "company",
                 linkedGauss = "consistent")
print(a)  

# Alternatively, SuppressDominantCells() can be run directly using the linkedGauss parameter  
a1 <- SuppressDominantCells(SSBtoolsData("magnitude1"), 
               formula = list(table_1 = ~(geo + eu) * sector2, 
                              table_2 = ~eu:sector4 - 1,
                              table_3 = ~(geo + eu) + sector4 - 1), 
               dominanceVar = "value", 
               pPercent = 10, 
               contributorVar = "company", 
               linkedGauss = "consistent")
print(a1)

# In fact, tables_by_formulas() is also a possibility
a2 <- tables_by_formulas(SSBtoolsData("magnitude1"),
               table_fun = SuppressDominantCells, 
               table_formulas = list(table_1 = ~region * sector2, 
                                    table_2 = ~region1:sector4 - 1, 
                                    table_3 = ~region + sector4 - 1), 
               substitute_vars = list(region = c("geo", "eu"), region1 = "eu"), 
               collapse_vars = list(sector = c("sector2", "sector4")), 
               dominanceVar  = "value", 
               pPercent = 10, 
               contributorVar = "company",
               linkedGauss = "consistent") 
print(a2)                 
               
               
               
               
####  The second example cannot be handled using the alternative methods.
####  This is similar to the (old) LazyLinkedTables() example.

z1 <- SSBtoolsData("z1")
z2 <- SSBtoolsData("z2")
z2b <- z2[3:5]  # As in ChainedSuppression example 
names(z2b)[1] <- "region" 
# As 'f' and 'e' in ChainedSuppression example. 
# 'A' 'annet'/'arbeid' suppressed in b[[1]], since suppressed in b[[3]].
b <- SuppressLinkedTables(fun = SuppressSmallCounts,
              linkedGauss = "consistent",  
              recordAware = FALSE,
              withinArg = list(
                list(data = z1, dimVar = 1:2, freqVar = 3, maxN = 5), 
                list(data = z2b, dimVar = 1:2, freqVar = 3, maxN = 5), 
                list(data = z2, dimVar = 1:4, freqVar = 5, maxN = 1)))
print(b)        
       
}
