
#' Number of contributors suppression rule
#' 
#' The number of contributors is the number unique contributing 'charVar' codes. 
#' 
#'
#' @param data  Input data as a data frame
#' @param freq  Vector of aggregate frequencies 
#' @param numVar Numerical variables. When several variables, only first is used. 
#' @param x Model matrix generated by parent function
#' @param maxN Primary suppression when number of contributors `<= maxN`.
#' @param protectZeros Suppression parameter. Only TRUE (default) is used implemented. 
#' @param charVar Variable with contributor codes. 
#' @param removeCodes Vector of codes to be omitted when counting contributors. 
#' @param remove0 When `FALSE` (default), data rows where `numVar` (if any) is zero are omitted when counting contributors.
#' @param ... unused parameters
#'
#' @return List where first element is logical vector defining primary suppressions.
#'         The second element is data frame where `nRule` is number contributors used 
#'         in rule and where `nAll` is similar, but without omitting codes in `removeCodes`. 
#' @export
#'
NcontributorsRule <- function(data, freq, numVar, x, maxN = 3, protectZeros = FALSE, charVar, removeCodes = character(0), remove0 = TRUE, ...) {
  if (length(charVar)>1) {
    stop("Only single charVar implemented in suppression rule")
  }
  if (length(numVar) > 1){
    warning("Multiple numVar were supplied, only the first is used in suppression rule.")
    numVar <- numVar[1]
  }
  if (protectZeros) {
    stop("TRUE protectZeros not implemented")
  }
  y <- data[[charVar]]
  if (remove0 & length(numVar) > 0) {
    y[data[[numVar]] == 0] <- NA
  }
  
  nAll <- Ncontributors(x, y)
  y[y %in% removeCodes] <- NA
  nRule <- Ncontributors(x, y)
  primary <- (nRule <= maxN) & (nRule > 0)
  list(primary = primary, numExtra = data.frame(nRule = nRule, nAll = nAll))
}




