#' gsnImportGSNORA
#'
#' @description Add GSNORA search data to a GSNData object, as generated by the \code{gsnORAtest} function in this package.
#' The data set can be either in the form of a data.frame or specified as import from a delimited text file.
#'
#' @param object A GSNData object.
#' @param pathways_data An (optional) data.frame containing the results of GSNORA analysis. (Either this or the
#' \code{filename} argument must be set.
#' @param filename An (optional) filename for data sets read from a text file containing GSNORA results. This is ignored
#' if the \code{pathways_data} argument is set.
#' @param id_col (optional) A character vector of length 1 indicating the name of the column used as a key for gene
#' sets or modules. This is normally the \code{ID} field of GSNORA data, which must be the same as the names of gene sets
#' specified in the tmod object or in the list of gene set vectors specified with the \code{geneSetCollection} argument
#' used when building the gene set network. By default this value is \code{'ID'}, however if the user has added additional
#' IDs to a CERNO results set, such as \code{GO_ACCESSION}, that can be specified here. The IDs must correspond to the names
#' of the gene sets provided, or an error will be thrown.
#' @param stat_col (optional) A character vector of length 1 indicating the name of the column used as a statistic
#' to evaluate the quality of pathways results. By default, this is \code{'adj.P.1S'} for GSNORA.
#' @param sig_order (optional) Either \code{'loToHi'} (default) or \code{'hiToLo'} depending on the statistic used to
#' evaluate pathways results.
#' @param n_col (optional) Specifies the column containing the number of genes in the gene set. Generally, this is the number
#' of genes in the gene set that are attested in an expression data set (Defaults to 'N').
#' @param sep A separator for text file import, defaults to "\\t". Ignored if the \code{filename} argument is not specified.
#'
#' @return This returns a GSNData object containing imported pathways data.
#'
#' Note: An error is thrown if all gene set IDs in the genePresenceAbsense are not present in the GSNORA ID column.
#' On the other hand, if there are gene set IDs present in the pathways data that are absent from the genePresenceAbsence
#' matrix, then these methods emit a warning. It also checks for the standard GSNORA data set column names, and if some are
#' missing, it will throw an error.
#'
#' @export
#'
#' @examples
#'
#' # In this example, we run gsnORAtest() to generate a list of significant
#' # pathways, which we use to construct a gene set network.
#'
#' library(GSNA)
#'
#' # From a differential expression data set, we can generate a subset of genes
#' # with significant differential expression, up or down. Here we will extract
#' # genes with significant negative differential expression with
#' # avg_log2FC < 0 and p_val_adj <= 0.05 from **Seurat** data:
#'
#' sig_DN.genes <-
#'    toupper( rownames(subset( Bai_CiHep_v_Fib2.de,
#'                              avg_log2FC < 0  & p_val_adj < 0.05 )) )
#'
#' # Using all the genes in the differential expression data set, we can obtain
#' # a suitable background:
#' bg <- toupper(rownames( Bai_CiHep_v_Fib2.de ))
#'
#' # Now, we can do a overrepresentation analysis search on this data using the
#' # Bai_gsc.tmod gene set collection included in the sample data:
#' sig_DN.gsnora <- gsnORAtest( l = sig_DN.genes,
#'                              bg = bg,
#'                              geneSetCollection = Bai_gsc.tmod )
#'
#' # Generate a GSC (gene set collection) from the significant gene sets:
#' sig.gsnora.tmod <- Bai_gsc.tmod[sig_DN.gsnora$ID]
#'
#' bg <- toupper( rownames( Bai_CiHep_v_Fib2.de ) )
#'
#' # Build gene set network from the GSC
#' sig.gsnora.GSN <- buildGeneSetNetworkSTLF( ref.background =  bg,
#'                                            geneSetCollection = sig.gsnora.tmod )
#'
#' sig.gsnora.GSN <- gsnImportGSNORA( sig.gsnora.GSN, sig_DN.gsnora )
#'
#'
#' @seealso
#'  \code{\link{gsnAddPathwaysData}}
#'  \code{\link{gsnImportCERNO}}
#'  \code{\link{gsnImportGSEA}}
#'  \code{\link{gsnImportGenericPathways}}
#'
#' @importFrom utils read.table
#'
gsnImportGSNORA <- function( object, pathways_data = NULL, filename = NULL, id_col = NULL, stat_col = NULL, sig_order = NULL, n_col = NULL, sep = "\t" ){
  stopifnot( "GSNData" %in% class( object ) )

  if( is.null( pathways_data ) && is.null( filename ) ) stop( "The 'pathways_data' and 'filename' arguments cannot both be NULL." )
  if( is.null( pathways_data ) ){
    pathways_data <- utils::read.table( file = filename, header = TRUE, sep = sep, stringsAsFactors = FALSE )
  }
  if( !is.null(sig_order) && ! sig_order %in% c( "loToHi", "hiToLo" ) )
    stop( "Invalid sig_order: ", as.character( sig_order ) )
  if( ! is.null(stat_col) && ! stat_col %in% colnames( pathways_data ) )
    stop( "stat_col '", stat_col, "' not found in pathways data."  )

  gsnora_fieldnames <- c("ID", "Title", "Enrichment", "P.1S", "adj.P.1S" )

  if( length( missing_fieldnames <- gsnora_fieldnames[! gsnora_fieldnames %in% colnames(pathways_data)] ) > 0 ){
    warning( "Data is missing the following GSNORA fields:", paste0( missing_fieldnames,  collapse = ", " ) )
  }

  pathways <- list( data = pathways_data, type = "gsnora", id_col = "ID", stat_col = "adj.P.1S", sig_order = "loToHi", n_col = "N" )
  if( !is.null(id_col) ) pathways$id_col <- id_col
  if( !is.null(stat_col) ) pathways$stat_col <- stat_col
  if( !is.null(sig_order) ) pathways$sig_order <- sig_order
  if( !is.null(n_col) ) pathways$n_col <- n_col

  if( ! all( colnames( object$genePresenceAbsence ) %in% pathways$data[[pathways$id_col]] ) )
    stop("Error: Pathways data do not match gene set collection. They are missing gene sets from gene set collection.")
  if( ! all( pathways$data[[pathways$id_col]] %in% colnames( object$genePresenceAbsence ) ) )
    warning("Warning: Pathways data do not match gene set collection. They contain gene sets not present in gene set collection.")

  object$pathways <- pathways
  object
}

