% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SparseGRM.R
\name{getSparseGRM}
\alias{getSparseGRM}
\title{Make a \code{SparseGRMFile} for \code{\link{GRAB.NullModel}}.}
\usage{
getSparseGRM(
  PlinkFile,
  nPartsGRM,
  SparseGRMFile,
  tempDir = NULL,
  relatednessCutoff = 0.05,
  minMafGRM = 0.01,
  maxMissingGRM = 0.1,
  rm.tempFiles = FALSE
)
}
\arguments{
\item{PlinkFile}{a path to PLINK binary files (without file extension). Note that the current version (gcta_1.93.1beta) of \code{GCTA} software does not support different prefix names for BIM, BED, and FAM files.}

\item{nPartsGRM}{a numeric value (e.g. 250): \code{GCTA} software can split subjects to multiple parts. For UK Biobank data analysis, it is recommended to set \code{nPartsGRM=250}.}

\item{SparseGRMFile}{a path to file of output to be passed to \code{\link{GRAB.NullModel}}.}

\item{tempDir}{a path to store temp files from \code{\link{getTempFilesFullGRM}}. This should be consistent to the input of \code{\link{getTempFilesFullGRM}}. Default is \code{system.file("SparseGRM", "temp", package = "GRAB")}.}

\item{relatednessCutoff}{a cutoff for sparse GRM, only kinship coefficient greater than this cutoff will be retained in sparse GRM. \emph{(default=0.05)}}

\item{minMafGRM}{Minimal value of MAF cutoff to select markers (from PLINK files) to make sparse GRM. \emph{(default=0.01)}}

\item{maxMissingGRM}{Maximal value of missing rate to select markers (from PLINK files) to make sparse GRM. \emph{(default=0.1)}}

\item{rm.tempFiles}{a logical value indicating if the temp files generated in \code{\link{getTempFilesFullGRM}} will be deleted. \emph{(default=FALSE)}}
}
\value{
A character string containing a message with the path to the output file
where the sparse Genetic Relationship Matrix (SparseGRM) has been stored.
}
\description{
If the sample size in analysis is greater than 100,000, we recommend using sparse GRM (instead of dense GRM) to adjust for sample relatedness.
This function is to use \code{GCTA} (\href{https://cnsgenomics.com/software/gcta/#Overview}{link}) to make a \code{SparseGRMFile} to be passed to function \code{\link{GRAB.NullModel}}.
This function can only support \code{Linux} and \code{PLINK} files as required by \code{GCTA} software. To make a \code{SparseGRMFile}, two steps are needed. Please check \code{Details} section for more details.
}
\details{
\itemize{
\item \code{Step 1}: Run \code{\link{getTempFilesFullGRM}} to save temporary files to \code{tempDir}.
\item \code{Step 2}: Run \code{getSparseGRM} to combine the temporary files to make a \code{SparseGRMFile} to be passed to function \code{\link{GRAB.NullModel}}.
}
Users can customize parameters including \code{(minMafGRM, maxMissingGRM, nPartsGRM)}, but functions \code{\link{getTempFilesFullGRM}} and \code{getSparseGRM} should use the same ones.
Otherwise, package \code{GRAB} cannot accurately identify temporary files.
}
\section{The following shows a typical workflow for creating a sparse GRM:}{
\code{# Input data (We recommend setting nPartsGRM=250 for UKBB with N=500K):}

\code{GenoFile = system.file("extdata", "simuPLINK.bed", package = "GRAB")}

\code{PlinkFile = tools::file_path_sans_ext(GenoFile)}

\code{nPartsGRM = 2}
}

\section{Step 1: We strongly recommend parallel computing in high performance clusters (HPC).}{
\code{# For Linux, get the file path of gcta64 by which command:}

\code{gcta64File <- system("which gcta64", intern = TRUE)}

\code{# For Windows, set the file path directly:}

\code{gcta64File <- "C:\\\\path\\\\to\\\\gcta64.exe"}

\code{# The temp outputs (may be large) will be in system.file("SparseGRM", "temp", package = "GRAB") by default:}

\code{for(partParallel in 1:nPartsGRM) getTempFilesFullGRM(PlinkFile, nPartsGRM, partParallel, gcta64File)}
}

\section{Step 2: Combine files in Step 1 to make a SparseGRMFile}{
\code{tempDir = system.file("SparseGRM", "temp", package = "GRAB")}

\code{SparseGRMFile = gsub("temp", "SparseGRM.txt", tempDir)}

\code{getSparseGRM(PlinkFile, nPartsGRM, SparseGRMFile)}
}

