% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/netest.R
\name{netest}
\alias{netest}
\title{Dynamic Network Model Estimation}
\usage{
netest(
  nw,
  formation,
  target.stats,
  coef.diss,
  constraints,
  coef.form = NULL,
  edapprox = TRUE,
  set.control.ergm = control.ergm(),
  set.control.tergm = control.tergm(),
  set.control.ergm.ego = control.ergm.ego(),
  verbose = FALSE,
  nested.edapprox = TRUE,
  ...
)
}
\arguments{
\item{nw}{An object of class \code{network} or \code{egor}, with the latter
indicating an \code{ergm.ego} fit.}

\item{formation}{Right-hand sided STERGM formation formula in the form
\code{~edges + ...}, where \code{...} are additional network
statistics.}

\item{target.stats}{Vector of target statistics for the formation model, with
one number for each network statistic in the model.  Ignored if
fitting via \code{ergm.ego}.}

\item{coef.diss}{An object of class \code{disscoef} output from the
\code{\link{dissolution_coefs}} function.}

\item{constraints}{Right-hand sided formula specifying constraints for the
modeled network, in the form \code{~...}, where \code{...} are
constraint terms. By default, no constraints are set.}

\item{coef.form}{Vector of coefficients for the offset terms in the formation
formula.}

\item{edapprox}{If \code{TRUE}, use the indirect edges dissolution
approximation  method for the dynamic model fit, otherwise use the
more time-intensive full STERGM estimation (see details).  For
\code{nw} of class \code{egor}, only \code{edapprox = TRUE} is
supported.}

\item{set.control.ergm}{Control arguments passed to \code{ergm} (see
details).}

\item{set.control.tergm}{Control arguments passed to \code{tergm}
(see details).}

\item{set.control.ergm.ego}{Control arguments passed to \code{ergm.ego} (see
details).}

\item{verbose}{If \code{TRUE}, print model fitting progress to console.}

\item{nested.edapprox}{Logical. If \code{edapprox = TRUE} the dissolution
model is an initial segment of the formation model (see details).}

\item{...}{Additional arguments passed to other functions.}
}
\value{
A fitted network model object of class \code{netest}.
}
\description{
Estimates statistical network models using the exponential
random graph modeling (ERGM) framework with extensions for
dynamic/temporal models (STERGM).
}
\details{
\code{netest} is a wrapper function for the \code{ergm}, \code{ergm.ego},
and \code{tergm} functions that estimate static and dynamic network models.
Network model estimation is the first step in simulating a stochastic
network epidemic model in \code{EpiModel}. The output from \code{netest} is
a necessary input for running the epidemic simulations in
\code{\link{netsim}}. With a fitted network model, one should always first
proceed to model diagnostics, available through the \code{\link{netdx}}
function, to check model fit. A detailed description of fitting these
models, along with examples, may be found in the
\href{http://www.epimodel.org/tut.html}{Basic Network Models} tutorials.
}
\section{Edges Dissolution Approximation}{

The edges dissolution approximation method is described in Carnegie et al.
This approximation requires that the dissolution coefficients are known, that
the formation model is being fit to cross-sectional data conditional on those
dissolution coefficients, and that the terms in the dissolution model are a
subset of those in the formation model. Under certain additional conditions,
the formation coefficients of a STERGM model are approximately equal to the
coefficients of that same model fit to the observed cross-sectional data as
an ERGM, minus the corresponding coefficients in the dissolution model. The
approximation thus estimates this ERGM (which is typically much faster than
estimating a STERGM) and subtracts the dissolution coefficients.

The conditions under which this approximation best hold are when there are
few relational changes from one time step to another; i.e. when either
average relational durations are long, or density is low, or both.
Conveniently, these are the same conditions under which STERGM estimation is
slowest. Note that the same approximation is also used to obtain starting
values for the STERGM estimate when the latter is being conducted. The
estimation does not allow for calculation of standard errors, p-values, or
likelihood for the formation model; thus, this approach is of most use when
the main goal of estimation is to drive dynamic network simulations rather
than to conduct inference on the formation model. The user is strongly
encouraged to examine the behavior of the resulting simulations to confirm
that the approximation is adequate for their purposes. For an example, see
the vignette for the package \code{tergm}.

It has recently been found that subtracting a modified version of the
dissolution coefficients from the formation coefficients provides a more
principled approximation, and this is now the form of the approximation
applied by \code{netest}. The modified values subtracted from the formation
coefficients are equivalent to the (crude) dissolution coefficients with
their target durations increased by 1. The \code{nested.edapprox} argument
toggles whether to implement this modified version by appending the
dissolution terms to the formation model and appending the relevant values to
the vector of formation model coefficients (value = \code{FALSE}), whereas
the standard version subtracts the relevant values from the initial formation
model coefficients (value = \code{TRUE}).
}

\section{Control Arguments}{

The \code{ergm}, \code{ergm.ego}, and \code{tergm} functions allow control
settings for the model fitting process. When fitting a STERGM directly (setting
\code{edapprox} to \code{FALSE}), control parameters may be passed to the
\code{tergm} function with the \code{set.control.tergm} argument in
\code{netest}. The controls should be input through the
\code{control.tergm()} function, with the available parameters listed in the
\code{\link{control.tergm}} help page in the \code{tergm} package.

When fitting a STERGM indirectly (setting \code{edapprox} to \code{TRUE}),
control settings may be passed to the \code{ergm} function using
\code{set.control.ergm}, or to the \code{ergm.ego} function using
\code{set.control.ergm.ego}.  The controls should be input through the
\code{control.ergm()} and \code{control.ergm.ego()} functions, respectively,
with the available parameters listed in the
\code{\link[ergm:control.ergm]{control.ergm}} help page in the \code{ergm}
package and the \code{\link[ergm.ego:control.ergm.ego]{control.ergm.ego}}
help page in the \code{ergm.ego} package. An example is below.
}

\examples{
# Initialize a network of 100 nodes
nw <- network_initialize(n = 100)

# Set formation formula
formation <- ~edges + concurrent

# Set target statistics for formation
target.stats <- c(50, 25)

# Obtain the offset coefficients
coef.diss <- dissolution_coefs(dissolution = ~offset(edges), duration = 10)

# Estimate the STERGM using the edges dissolution approximation
est <- netest(nw, formation, target.stats, coef.diss,
              set.control.ergm = control.ergm(MCMC.burnin = 1e5,
                                              MCMC.interval = 1000))
est

# To estimate the STERGM directly, use edapprox = FALSE
# est2 <- netest(nw, formation, target.stats, coef.diss, edapprox = FALSE)

}
\references{
Krivitsky PN, Handcock MS. "A separable model for dynamic networks." JRSS(B).
2014; 76.1:29-46.

Carnegie NB, Krivitsky PN, Hunter DR, Goodreau SM. An approximation method
for improving dynamic network model fitting. Journal of Computational and
Graphical Statistics. 2014; 24(2): 502-519.

Jenness SM, Goodreau SM and Morris M. EpiModel: An R Package for Mathematical
Modeling of Infectious Disease over Networks. Journal of Statistical
Software. 2018; 84(8): 1-47.
}
\seealso{
Use \code{\link{netdx}} to diagnose the fitted network model, and
\code{\link{netsim}} to simulate epidemic spread over a simulated
dynamic network consistent with the model fit.
}
\keyword{model}
