% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eco.malecot.R
\name{eco.malecot}
\alias{eco.malecot}
\title{Global and local kinship analysis}
\usage{
eco.malecot(
  eco,
  method = c("global", "local"),
  kinmatrix = NULL,
  int = NULL,
  smin = 0,
  smax = NULL,
  nclass = NULL,
  kmax = NULL,
  seqvec = NULL,
  size = NULL,
  type = c("knearest", "radialdist"),
  cubic = TRUE,
  testclass.b = TRUE,
  testmantel.b = TRUE,
  jackknife = TRUE,
  cummulative = FALSE,
  normLocal = TRUE,
  nsim = 99,
  test = c("permutation", "bootstrap"),
  alternative = c("auto", "two.sided", "greater", "less"),
  sequential = TRUE,
  conditional = c("AUTO", "TRUE", "FALSE"),
  bin = c("sturges", "FD"),
  row.sd = FALSE,
  adjust = "holm",
  latlon = FALSE,
  angle = NULL
)
}
\arguments{
\item{eco}{Object of class ecogen.}

\item{method}{Analysis method: "global" or "local".}

\item{kinmatrix}{Alternative kinship matrix. The program computes
the Loiselle's kinship matrix (codominant data) with the genetic information
of the ecogen object if kinmatrix = NULL (Defaul option).}

\item{int}{Distance interval in the units of XY.}

\item{smin}{Minimum class distance in the units of XY.}

\item{smax}{Maximum class distance in the units of XY.}

\item{nclass}{Number of classes.}

\item{kmax}{Number of nearest-neighbors for local analysis.}

\item{seqvec}{Vector with breaks in the units of XY.}

\item{size}{Number of individuals per class.}

\item{type}{Weighting mode for local analysis: "knearest" for nearest neigbors,
"radialdist" for radial distances. Default is knearest.}

\item{cubic}{Should a cubic interpolation (res~ ln(dij)) be performed, 
for the regression residuals (res)  of (kinship)ij ~ ln(dij) ? Default TRUE.}

\item{testclass.b}{Carry a permutation test within each individual class? Default TRUE.}

\item{testmantel.b}{Should a Mantel test for testing the slope (b) be performed? Default TRUE.}

\item{jackknife}{Compute jackknife within each individual class for obtention of
the standard deviation (SD) of the coancestry (class) values. Default TRUE.}

\item{cummulative}{Should a cummulative correlogram be construced?.}

\item{normLocal}{Normalize the local kinship values ([local_kinship-mean]/sd)? Default TRUE}

\item{nsim}{Number of Monte-Carlo simulations.}

\item{test}{If test = "bootstrap", the program generates a bootstrap 
resampling and the associated confidence intervals of the null hypothesis.
 If test = "permutation" (default) a permutation test is made and the P-values 
 are computed.}

\item{alternative}{The alternative hypothesis. If "auto" is selected (default) the
program determines the alternative hypothesis.
Other options are: "two.sided", "greater" and "less".}

\item{sequential}{Use the Holm-Bonberroni sequential method for
adjustment of P-values (Legendre and Legendre, 2012) in global analysis? Default TRUE.}

\item{conditional}{Logical. Use a conditional randomization? (Anselin 1998, Sokal and Thomson 2006). The option "auto"
sets conditional = TRUE for LISA methods and G, as suggested by Sokal (2008).}

\item{bin}{Rule for constructing intervals when a partition parameter (int, 
nclass or size) is not given. Default is Sturge's rule (Sturges, 1926). Other
option is Freedman-Diaconis method (Freedman and Diaconis, 1981).}

\item{row.sd}{Logical. Should be row standardized the matrix? Default FALSE 
(binary weights).}

\item{adjust}{P-values correction method for multiple tests 
passed to \code{\link[stats]{p.adjust}}. Default is "holm".}

\item{latlon}{Are the coordinates in decimal degrees format? Default FALSE. If TRUE,
the coordinates must be in a matrix/data frame with the longitude in the first
column and latitude in the second. The position is projected onto a plane in
meters with the function \code{\link[SoDA]{geoXY}}.}

\item{angle}{direction for computation of a bearing correlogram (angle in degrees between 0 and 180).
Default NULL (omnidirectional).}
}
\value{
For the global analysis, the program returns an object of class "eco.IBD" 
with the following slots:

> OUT analysis output. 


In the permutation test case contains: 
- d.mean: mean class distance;
- d.log: mean logarithm of the class distance;
- obs, exp, alter, p.val: observed, and expected value of the statistic
under randomization, alternative, P value;
- mean.jack, sd.jack, Jack.CI.inf, Jack.CI.sup: jackknifed mean and SD,
and confidence intervals for the statistic;
- null.lwr, nul.uppr: lower and upper bound of the jackknife
 confidence interval for the statistic;
- cardinal: number of individuals in each class;


In the bootstrap test case contains: 
- d.mean: mean class distance;
- d.log: mean logarithm of the class distance;
- obs: observed value of the statistic;
- mean.jack, sd.jack, Jack.CI.inf, Jack.CI.sup: jackknifed mean and SD,
and confidence intervals for the statistic;
- null.lwr, nul.uppr: lower and upper bound of the jackknife
 confidence interval for the statistic;
- cardinal: number of individuals in each class;

> GLOBALTEST Oden's (1984) global test of significance for the correlogram.
The test consists in checking if the most significant kinship coefficent is 
significant at a Bonferroni- corrected significance level of alpha' = alpha/k, where
k is the number of distance classes of the correlogram; alpha is set to 0.05.
The program return the values: "SIGNIFICANT" or "NOT-SIGNIFICANT"

> IN analysis input data

> SP Sp statistic results


It contains:

- the regression model;
- information about the distance interval used 
for the regression (restricted);
-  slope (bhat) information (bhat = estimate, SD= bhat jacknife SD, theta =  bhat jackknife mean, 
CI 5\% and 95\% = 95\% confidence interval for bhat);
-  X-intercept = dij intercept (in the original units) for the line with slope "bhat", 
F1 = first class statistic value, and  F1 5\% and 95\% = confidence interval
for the first class statistic;
- mantel.obs.b = observed value of the Mantel test between
kinship(Fij) and ln(dij); mantel.pval.b = Mantel test P value;
- sp = Sp statistics (sp = Sp observed value, 
CI 5\% and 95\% = 95\% confidence interval 
for Sp);
- cubic_model = cubic model for (kinship)ij ~ ln(dij) r
esiduals vs ln(dij);

> BEAKS breaks

> CARDINAL number of elements in each class

> NAMES variables names

> METHOD analysis method

> DISTMETHOD method used in the construction of breaks

> TEST test method used (bootstrap, permutation)

> NSIM number of simulations

> PADJUST P-values adjust method for permutation tests

------

For the local analysis, the program returns an object of class "eco.lsa" 
with the following slots:

> OUT results


> In the permutation test case it contains: 

- d.mean: mean class distance
- obs, exp, alter, p.val: observed, and expected value of the statistic
under randomization, alternative, P value;
- null.lwr, nul.uppr: lower and upper bound of the jackknife
 confidence interval for the statistic;
- cardinal: number of individuals in each class;


> In the bootstrap test case it contains: 
- d.mean: mean class distance;
- obs: observed value of the statistic;
- null.lwr, nul.uppr: lower and upper bound of the jackknife;
 confidence interval for the statistic;
- cardinal: number of individuals in each class;

> METHOD method (coefficent) used in the analysis

> TEST test method used (bootstrap, permutation)

> NSIM number of simulations

> PADJUST P-values adjust method for permutation tests

> COND conditional randomization (logical)

> XY input coordinates 


\strong{ACCESS TO THE SLOTS}
The content of the slots can be accessed 
with the corresponding accessors, using
the generic notation of EcoGenetics 
(<ecoslot.> + <name of the slot> + <name of the object>).
See help("EcoGenetics accessors") and the Examples
section below.
}
\description{
The program computes, for a kinship matrix,  a global multilocus correlogram,
 or a local analysis. When a kinship matrix is not given as input, the program
computes the Loiselle's Fij (Kalisz et al., 2001; Loiselle et al., 1995). 
The program can compute a bearing correlogram (Rosenberg 2000, Born et al.
2012) for the obtention of a directional approach in the global test.
}
\details{
The GLOBAL ANALYSIS mode, computes a multilocus correlogram, with a detailed
summary (see the content of the slot OUT  in the "return" section). 
It also computes (see details about the slot SP in the "return" section):
- the slope of the kinship individual values vs the logarithm of the distance, (kinship)ij ~ ln(dij), with a jackknife confidence
interval 
- a Mantel test for testing the association between  (kinship)ij
and ln(dij) 
- The Sp statistic (Vekemans and Hardy, 2004) with confidence intervals
- A cubic interpolation of (kinship)ij ~ ln(dij) residuals vs ln(dij)

A directional approach is based on the bearing analysis method, and consists in the 
obtention of a directional correlogram using the method of Rosenberg (2000). A
slope is computed for the logarithm of D' (Born et al 2012), where D' is the distance matrix
between individuals weighted by cos(alpha - B)^2, being alpha the angle
between individuals and B the desired direction angle. With B = 0
the direcction analyzed follows the positive x axis, with B = 0 the positive y axis,
 and with B = 180 the negative x axis, respectively.


The LOCAL ANALYSIS mode, computes a local kinship estimate, based in a weighted 
mean (for each individual). The significance of each local statistic
is computed using a permutation test, as in eco.lsa (see ?"eco.lsa"). 
Default option do not adjust the individual P values 
for multiple comparisons.
}
\examples{

\dontrun{

data(eco.test)

# ---global analysis---

globaltest <- eco.malecot(eco=eco, method = "global", smax=10,
                         size=1000)
eco.plotCorrelog(globaltest)    # Significant mean class coancestry classes at   
                    # individual level (alpha = 0.05, 
                    # out of the red area), 
                    # and family-wise P corrected values (red-blue
                    # points, indicated in the legend)

# ecoslot.SP(globaltest) contains:
# - the slope (bhat) and values with confidence intervals
#  of the regression reg = kinship ~ ln(distance_between_individuals)
#- A Mantel test result for assesing the relation between
#  between  kinship and ln(distance_between_individuals)
#- A cubic interpolation between the residuals of reg and 
#  ln(distance_between_individuals)
#- the sp statistic and its confidence interval

# ecoslot.OUT(globaltest) contains:
# - In permutation case, the values of mean and log-mean distance    
#   classes; observed class value; expected + alternative + P value,
#   the bootstrap null confidence intervals and 
#   jackknife statistics (jackknifed mean, jackknifed SD, and
#                         CI for the class statistic)

# - In bootstrap case, the values of mean and log-mean distance
#   classes;the bootstrap null confidence intervals and 
#   jackknife statistics (jackknifed mean, jackknifed SD, and
#                         CI for the class statistic)


# A directional approach based in bearing correlograms, 30 degrees
globaltest_30 <- eco.malecot(eco=eco, method = "global", smax=10,
                         size=1000, angle = 30)
eco.plotCorrelog(globaltest) 

#----------------------------------------------------------#
# ---local analysis---


# (using the spatial weights). 

# ---local analysis with k nearest neighbors---



localktest <- eco.malecot(eco=eco, method = "local",
                         type = "knearest", kmax = 5, 
                         adjust = "none")
eco.plotLocal(localktest)


# ---local analysis with radial distance---

localdtest <- eco.malecot(eco=eco, method = "local",
                        type = "radialdist", smax = 3, 
                        adjust = "none")
                        
eco.plotLocal(localdtest)                    # rankplot graphic (see ?"eco.rankplot")

                                         # Significant values
                                         # in blue-red scale, 
                                         # non significant 
                                         # values in yellow

eco.plotLocal(localktest, significant = FALSE)    # significant and non
                                              # signficant values
                                              # in blue-red scale

# The slot OUT of localktest (ecoslot.OUT(localktest)) and localdtest 
# (ecoslot.OUT(localdtest)) contains:
# - the mean distance per individual, observed value of the
#   statistic, expected + alternative + P value + null hypotesis
#   confidence intervals,  or boostrap confidence intervals in 
#   permutation or bootstrap cases, respectively.
}

}
\references{
Born C., P. le Roux, C. Spohr, M. McGeoch, B. Van Vuuren. 2012.
Plant dispersal in the sub-Antarctic inferred from anisotropic genetic structure.
Molecular Ecology 21: 184-194.

Double M., R. Peakall, N. Beck, and Y. Cockburn. 2005. 
Dispersal, philopatry, and infidelity: dissecting 
local genetic structure in superb fairy-wrens (Malurs cyaneus). 
Evolution 59: 625-635.

Kalisz S., J. Nason, F.M. Handazawa, and S. Tonsor. 2001. 
Spatial population genetic structure in Trillium grandiflorum: 
the roles of dispersal, mating, history, and selection. 
Evolution 55: 1560-1568.

Loiselle B., V. Sork, J. Nason, and C. Graham. 1995. 
Spatial genetic structure of a tropical understory shrub, 
Psychotria officinalis (Rubiaceae). 
American Journal of Botany 1420-1425.

Oden, N., 1984. Assessing the significance of a spatial correlogram. 
Geographical Analysis, 16: 1-16.

Rosenberg, M. 2000. The bearing correlogram: a new method 
of analyzing directional spatial autocorrelation. 
Geographical Analysis, 32: 267-278.

Vekemans, X., and O. Hardy. 2004. New insights from fine-scale 
spatial genetic structure analyses in plant populations. 
Molecular Ecology, 13: 921-935.
}
\author{
Leandro Roser \email{learoser@gmail.com}
}
