% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DTDAcif.R
\name{DTDAcif}
\alias{DTDAcif}
\title{Doubly Truncated Data Analysis, Cumulative Incidence Functions}
\usage{
DTDAcif(x, u, v, comp.event, method = c("indep", "dep"), boot = F,
  B = 300, N.iter = 100, error = 1e-06)
}
\arguments{
\item{x}{Numeric vector corresponding to the variable of ultimate interest.}

\item{u}{Numeric vector corresponding to the left truncation variable.}

\item{v}{Numeric vector corresponding to the right truncation variable.}

\item{comp.event}{Competing risk indicator.}

\item{method}{The method used to compute the nonparametric estimator. Use ‘indep’ for independent truncation variables and ``dep`` for truncation variables possibly depending on the competing risk.}

\item{boot}{Logical. If TRUE the bootstrap standard deviation of the cumulative incidences is calculated.}

\item{B}{Number of bootstrap replicates.}

\item{N.iter}{Maximum number of iterations.}

\item{error}{Error criterion for convergence.}
}
\value{
A list containing:
\itemize{
 \item{method: }{The method used to compute the estimator.}
 \item{biasf: }{The biasing function which reports the sampling probability for each Xi.}
 \item{cif.mas: }{The mass attached to each (Xi,Zi). The cumsum of cif.mas for Zi=j is the estimator of the j-th cumulative incidence function.}
 \item{data: }{The data corresponding to (X,Z) ordered with respect to X within each Z-value.}
 \item{sd.boot: }{The bootstrap standard deviation.}
}
}
\description{
This function computes a nonparametric estimator of the cumulative incidences of competing risks under double truncation. The estimator generalizes the Efron-Petrosian NPMLE (Non-Parametric Maximun Likelihood Estimator) to the competing risks setting.
}
\details{
The nonparametric estimator is based on the Efron-Petrosian NPMLE (Efron and Petrosian, 1999).
 Actually, each pair (Xi,Zi) -where Xi stands for the variable of interest and Zi is the competing event
 indicator- is weighted by the jump of the Efron-Petrosian NPMLE at Xi (method=``indep"), or by a normalized
 version of the Efron-Petrosian NPMLE computed from the subset of (Xs,Zs)'s such that Zs=Zi (method=``dep'').
 The former is suitable when the truncating couple (U,V) is independent of (X,Z), while the latter is recommended
 when (U,V) and X are only conditionally independent given Z; see de Uña-Álvarez (2019) for a full description of
 the estimators and of their properties. When the competing event indicator is missing, the function simply computes
 the Efron-Petrosian NPMLE and the argument method has no role.
}
\section{Acknowledgements}{

\itemize{
\item{Jacobo de Uña-Álvarez was supported by Grant MTM2017-89422-P (MINECO/AEI/FEDER, UE).}
\item{José Carlos Soage was supported by Grupos de Referencia Competitiva,
 Consolidación y Estructuración de Unidades de Investigación Competitivas del SUG,
 Cons. de Cultura, Educación e OU, Xunta de Galicia (GRC ED431C 2016/040).}
}
}

\examples{

\dontshow{

set.seed(1234)
n <- 25  # sample size

x <- runif(n, 0, 1)  # time variable of interest
z <- rbinom(n, 1, 1 / 4)   # competing event indicator

# truncation variables

u <- runif(n, -.25, .5)  # left truncation variable
v <- u + .75   # right truncation variable

# note: (u,v) is independent of (x,z) so both estimation methods are consistent

# truncating the sample:

for (i in 1:n) {
  while (u[i] > x[i] | v[i] < x[i]) {
    x[i] <- runif(1, 0, 1)
    z[i] <- rbinom(1, 1, 1 / 4)
    u[i] <- runif(1, -.25, .5)
    v[i] <- u[i] + .75
  }
}

# note: (u,v) since is independent of (x,z)
# both estimation methods are consistent:

res.i <- DTDAcif(x, u, v, z, boot = TRUE)
}

\donttest{
set.seed(1234)
n <- 50  # sample size

x <- runif(n, 0, 1)  # time variable of interest
z <- rbinom(n, 1, 1 / 4)   # competing event indicator

# truncation variables

u <- runif(n, -.25, .5)  # left truncation variable
v <- u + .75   # right truncation variable

# note: (u,v) is independent of (x,z) so both estimation methods are consistent

# truncating the sample:

for (i in 1:n) {
  while (u[i] > x[i] | v[i] < x[i]) {
    x[i] <- runif(1, 0, 1)
    z[i] <- rbinom(1, 1, 1 / 4)
    u[i] <- runif(1, -.25, .5)
    v[i] <- u[i] + .75
  }
}

# note: (u,v) since is independent of (x,z)
# both estimation methods are consistent:

res.i <- DTDAcif(x, u, v, z, method = "indep", boot = TRUE)
res.d <- DTDAcif(x, u, v, z, method = "dep", boot = TRUE)

oldpar <- par(mfrow=c(1,2))
plot(res.i, main = "Indep trunc", intervals = TRUE)
plot(res.d, main = "Cond indep trunc", intervals = TRUE)

summary(res.i)
summary(res.d)

plot(res.i$data$x, res.i$biasf, type = "s")  # the observational bias
# the observational bias, event 1
plot(res.d$data$x[res.d$data$z == 1], res.d$biasf$biasf_1, type = "s")
# the observational bias, event 2
lines(res.d$data$x[res.d$data$z == 2], res.d$biasf$biasf_2, type = "s", col = 2)
par(oldpar)
}

}
\references{
\itemize{
\item{de Uña-Álvarez, J. (2019). Nonparametric estimation of the cumulative incidences of competing risks under double truncation. Preprint.}
\item{Efron, B. and Petrosian, V. (1999). Nonparametric methods for doubly truncated data. Journal of the American Statistical Association 94, 824-834.}
}
}
\author{
\itemize{
\item{de Uña-Álvarez, Jacobo.}
\item{Soage González, José Carlos.}
\item{Maintainer: José Carlos Soage González. \email{jsoage@uvigo.es}}
}
}
