% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/interface.R
\name{cvek}
\alias{cvek}
\title{Conducting Cross-validated Kernel Ensemble}
\usage{
cvek(
  formula,
  kern_func_list,
  data,
  formula_test = NULL,
  mode = "loocv",
  strategy = "stack",
  beta_exp = 1,
  lambda = exp(seq(-10, 5)),
  test = "boot",
  alt_kernel_type = "linear",
  B = 100,
  verbose = FALSE
)
}
\arguments{
\item{formula}{(formula) A user-supplied formula for the null model. Should 
contain at least one kernel term.}

\item{kern_func_list}{(list) A list of kernel functions in the kernel library.}

\item{data}{(data.frame, n*d) A data.frame, list or environment (or object
coercible by as.data.frame to a data.frame), containing the variables in
formula. Neither a matrix nor an array will be accepted.}

\item{formula_test}{(formula) A user-supplied formula indicating the alternative
effect to test. All terms in the alternative mode must be specified as kernel terms.}

\item{mode}{(character) A character string indicating which tuning parameter
criteria is to be used.}

\item{strategy}{(character) A character string indicating which ensemble
strategy is to be used.}

\item{beta_exp}{(numeric/character) A numeric value specifying the parameter
when strategy = "exp" \code{\link{ensemble_exp}}.}

\item{lambda}{(numeric) A numeric string specifying the range of 
tuning parameter to be chosen. The lower limit of lambda must be above 0.}

\item{test}{(character) Type of hypothesis test to conduct. Must be either 
'asymp' or 'boot'.}

\item{alt_kernel_type}{(character) Type of alternative kernel effect to consider.
Must be either "linear" or "ensemble".}

\item{B}{(numeric) Number of bootstrap samples.}

\item{verbose}{(logical) Whether to print additional messages.}
}
\value{
\item{lambda}{(numeric) The selected tuning parameter based on the
estimated ensemble kernel matrix.}

\item{beta}{(matrix, d_fixed*1) Fixed effect estimates.}

\item{alpha}{(matrix, n*1) Kernel effect estimates.}

\item{K}{(matrix, n*n) Estimated ensemble kernel matrix.}

\item{u_hat}{(vector of length K) A vector of weights of the kernels in the
library.}

\item{base_est}{(list) The detailed estimation results of K kernels.}

\item{pvalue}{(numeric) If formula_test is given, p-value of the test is returned.}
}
\description{
Conducting Cross-validated Kernel Ensemble based on user-specified formula.
}
\details{
Perform Cross-validated Kernel Ensemble and optionally test for kernel effect
based on user-specified formula.
}
\examples{

kern_par <- data.frame(method = rep("rbf", 3),
l = rep(3, 3), p = rep(2, 3), 
stringsAsFactors = FALSE)
# define kernel library
kern_func_list <- define_library(kern_par)

n <- 10
d <- 4
formula <- y ~ x1 + x2 + k(x3, x4)
formula_test <- y ~ k(x1, x2) * k(x3, x4)
data <- as.data.frame(matrix(
  rnorm(n * d),
  ncol = d,
  dimnames = list(NULL, paste0("x", 1:d))
))
beta_true <- c(1, .41, 2.37)
lnr_kern_func <- generate_kernel(method = "rbf", l = 3)
kern_effect_lnr <-
  parse_kernel_variable("k(x3, x4)", lnr_kern_func, data)
alpha_lnr_true <- rnorm(n)

data$y <- as.matrix(cbind(1, data[, c("x1", "x2")])) \%*\% beta_true +
  kern_effect_lnr \%*\% alpha_lnr_true

data_train <- data

pvalue <- cvek(formula,
               kern_func_list,
               data_train,
               formula_test,
               mode = "loocv",
               strategy = "stack",
               beta_exp = 1,
               lambda = exp(seq(-2, 2)),
               test = "asymp",
               alt_kernel_type = "linear",
               verbose = FALSE)$pvalue

}
\references{
Xihong Lin. Variance component testing in generalised linear
models with random effects. June 1997.

Arnab Maity and Xihong Lin. Powerful tests for detecting a gene effect in
the presence of possible gene-gene interactions using garrote kernel
machines. December 2011.

Petra Bu z kova, Thomas Lumley, and Kenneth Rice. Permutation and
parametric bootstrap tests for gene-gene and gene-environment interactions.
January 2011.
}
\seealso{
\code{\link{estimation}}

method: \code{\link{generate_kernel}}

mode: \code{\link{tuning}}

strategy: \code{\link{ensemble}}
}
\author{
Jeremiah Zhe Liu
}
