#' Analysing a Closed Testing Procedure
#'
#' Calculation of p-values of a closed testing procedure (CTP).
#'	The function returns an object of \code{oldClass "ctp"; summary()} and \code{Display()} can be applied.
#'
#'	@param ctp.struc
#'		Object generated by the function \code{\link{IntersectHypotheses}} (structure of CTP)
#'	
#'	@param model
#'		model of the form response~treatment. If \code{testname="F"}, the model can be extended by covariates and other factors.
#'		In the case
#'		of a Logrank test the response must be imputed as \code{Surv(time,status)}. 
#'	
#'	@param dataset
#'		Dataframe, missing values in the response or treatment variable are not allowed! 
#'	
#'	@param factor.name
#'		Character string naming the factor whose levels are compared (treatment factor).
#'		By default the first variable of the right-hand side of the model formula is used. 
#'	
#'	@param test.name
#'		One of the following strings \itemize{
#'			\item \code{"F"} - F-Test (ANOVA, default)
#'			\item \code{"kruskal"} -Kruskal-Wallis-Test
#'			\item \code{"chisq"} - Chi square test
#'			\item \code{"prob"} - Fisher's exact test for total number of observations <200 else Chi square test
#'			\item \code{"lgrank"} - Logrank-test 
#'			\item \code{"jonckheere"} - Jonckheere-Terpstra test of ordered alternatives
#'		}
#'	@param ... Additional arguments for the chosen test.
#'	
#'	@details
#'		The hypothesis tree of the closed testing procedure must be created using \code{\link{IntersectHypotheses}}. For more details on the theory and the implementation as well
#'		for many examples, see the vignettes.

#' @return
#'  An object of old class(\code{ctp}), consisting of a list with:
#'   \itemize{
#'    \item{CTPparms} List with objects describing the CTP setup.
#'    \item{pvalues} Dataframe with all tested hypotheses, raw and adjusted p-values.
#'	  }
#'
#'	@note
#'		This procedure is constructed for testing differences and two-sided hypotheses,
#'		but not for equivalence tests. It is further based on independent samples from the population involved
#'		(i.e. on parallel group designs, but not on cross-over designs). 
#'	
#'	
#'	@seealso
#'		\code{\link{IntersectHypotheses}}, \code{\link{Display}}, \code{\link{summary.ctp.str}},  \code{\link{summary.ctp}},
#'		\code{\link{Adjust_raw}}
#'
#'	@examples
#'			
#'		data(pasi)
#'		three.to.first <- IntersectHypotheses(list(1:2,c(1,3),c(1,4)))
#'		Display(three.to.first)
#'		pasi.ctp.F1 <- AnalyseCTP(three.to.first,pasi.ch~dose,pasi)
#'		summary(pasi.ctp.F1)
#'		Display(pasi.ctp.F1)
#'		
#' @export

AnalyseCTP <- function(ctp.struc, model, dataset, factor.name = NULL, test.name = "F",...)
	{
		
		if(!(oldClass(ctp.struc) == "ctp.str")) stop("First argument must be an object of class ctp.str")
		if(!(oldClass(model) == "formula")) stop("Second argument must be a formula")
		if(!(oldClass(dataset) == "data.frame"))
			stop("Third argument must be a data frame")
		
		tn <- 0.
		
		for(tname in c("F", "kruskal", "chisq", "prob", "lgrank", "jonckheere")) 
		     tn <- tn + (tname == test.name)
		if(tn < 1.) stop("test.name wrongly specified")
  
		 CTPparms <- getCTPparms(ctp.struc=ctp.struc, model=model, dataset=dataset, factor.name = factor.name, test.name = test.name)
     pvalues  <- CTPcompare(CTPparms=CTPparms,...)
     pvadj    <- Adjust_p(ctp.struc = ctp.struc, ctp.pval = pvalues)
		 ctp.res  <- list(CTPparms=CTPparms, pvalues = pvadj)
     
		 #cat("\nCOMPARISONS finished ADJUSTING p-values\n")
	   oldClass(ctp.res) <- "ctp"		 
		 #cat("\nSUCCESFULLY FINISHED, apply summary() or Display() to the resulting object!\n")
		 ctp.res

	}

