## Function to call CPBayes_cor (CPBayes function for correlated summary statistics)
#' Run correlated version of CPBayes.
#' 
#' Run correlated version of CPBayes when the main genetic effect (beta/log(odds ratio)) estimates across
#'  studies/traits are correlated.
#' @param BetaHat A numeric vector of dimension K where K is the number of phenotypes. It
#'  contains the beta-hat values across studies/traits. No default is specified.
#' @param SE A numeric vector with the same dimension as BetaHat providing the standard errors
#'  corresponding to BetaHat. Every element of SE must be positive. No default is specified.
#' @param Corln A numeric square matrix of order K by K providing the correlation matrix of BetaHat.
#'  The number of rows of Corln must be the same as the dimension of the BetaHat vector. No default
#'   is specified.
#' @param Phenotypes A character vector of the same dimension as BetaHat providing the name
#'  of the phenotypes. Default is specified as trait1, trait2, . . . , traitK. Note that
#'   BetaHat, SE, Corln, and Phenotypes must be in the same order.
#' @param Variant A character vector of length 1 providing the name of the genetic variant.
#'  Default is `Variant'.
#' @param UpdateDE A logical vector of length 1. Default is TRUE. If FALSE,
#'  the variance of the slab distribution is considered to be fixed at 1. If TRUE, it is
#'   updated at each MCMC iteration in a range (0.8-1.2).
#' @param MCMCiter A positive integer greater than or equal to 10,000. Default is 20,000.
#' @param Burnin A positive integer greater than or equal to 5,000. Default is 10,000.
#'  Note that the MCMC sample size (MCMCiter - Burnin) must be at least 5,000.
#' @return The output produced by \code{\link{cpbayes_cor}} is a list which consists of various components.
#'    \item{variantName}{It is the name of the genetic variant provided by the user. If 
#'    not specified by the user, default name is `variant'.} 
#'    \item{log10_BF}{It provides the log10(Bayes factor) produced by CPBayes that measures the
#'     evidence of the overall pleiotropic association.}
#'    \item{PPNA}{It provides the posterior probability of null association produced by
#'     CPBayes (a Bayesian analog of the p-value) which is another measure of the evidence
#'      of aggregate-level pleiotropic association.}
#'    \item{subset}{It provides the optimal subset of associated/non-null traits estimated by
#'     CPBayes. It is NULL if no phenotype is selected by CPBayes.}
#'    \item{important_traits}{It provides the traits which yield a trait-specific posterior 
#'    probability of association (PPAj) > 25\%. Even if a phenotype is not 
#'    selected in the optimal subset of non-null traits, it can produce a non-negligible
#'     value of PPAj. We note that `important_traits' is expected to include the traits already
#'      contained in `subset'. It provides both the name of the important traits and their
#'       corresponding value of PPAj. Always check out 'important_traits' even if 'subset' contains
#'         no trait or a single trait. It helps to better explain an observed pleiotropic signal.}
#'    \item{auxi_data}{It contains supplementary data generated by the MCMC which is used later by the
#'     \code{\link{post_summaries}} function to provide additional insights into a pleiotropic signal.
#'      The supplementary data contained in auxi_data are as follows.
#'        \enumerate{
#'            \item traitNames: Name of all the phenotypes.
#'            \item K: Total number of phenotypes.
#'            \item mcmc.samplesize: MCMC sample size.
#'            \item asso.pr: Trait-specific posterior probability for all the traits. 
#'            \item Z.data: MCMC data on the latent association status (Z).
#'            \item sim.beta: MCMC data on the unknown true genetic effect (beta) on each trait.
#'        }
#'    }
#'    \item{uncor_use}{'Yes' or 'No'. Whether the combined strategy of CPBayes (implemented for correlated
#'     summary statistics) used the uncorrelated version or not.}
#'    \item{runtime}{It provides the runtime (in seconds) taken by CPBayes. It will help the user to
#'     plan the whole analysis.}
#'    
#' @references Arunabha Majumdar, Tanushree Haldar, Sourabh Bhattacharya, John Witte.
#'  An efficient Bayesian meta-analysis 
#'  approach for studying cross-phenotype genetic associations (submitted). Available
#'  at: http://biorxiv.org/content/early/2017/01/18/101543.
#'    
#' @seealso  \code{\link{estimate_corln}}, \code{\link{post_summaries}}, \code{\link{cpbayes_uncor}}
#' 
#' @examples
#' data(ExampleDataCor)
#' BetaHat <- ExampleDataCor$BetaHat
#' BetaHat
#' SE <- ExampleDataCor$SE
#' SE
#' cor <- ExampleDataCor$cor
#' cor
#' traitNames <- paste("Disease", 1:10, sep = "")
#' SNP1 <- "rs1234"
#' result <- cpbayes_cor(BetaHat, SE, cor, Phenotypes = traitNames, Variant = SNP1)
#' str(result)
#' 
#' 
#' @export
cpbayes_cor <- function(BetaHat, SE, Corln, Phenotypes, Variant, UpdateDE = TRUE, MCMCiter = 20000, Burnin = 10000)
{
  # Check whether any of the primary arguments is missing
  if(missing(BetaHat) || missing (SE))
    stop("BetaHat or SE vector is missing!", call. = FALSE)
  if(missing(Corln))
    stop("Correlation matrix is missing!", call. = FALSE)
  # Argument 1 :: BetaHat
    BetaHat <- checkPrimaryVar(BetaHat, "BetaHat")
  # Argument 2 :: SE
    SE <- checkPrimaryVar(SE, "SE")
  # Check whether all entries are strictly positive
  if(!all(SE > 0))
    stop("One or more elements in the SE vector are not positive!", call. = FALSE)
  # Argument 1 and 2 ::
  if(length(BetaHat) != length(SE))
    stop("BetaHat and SE vectors must have the same number of elements!", call. = FALSE)
  
  # Argument 3 :: Correlation
    COR <- checkCorln(Corln, BetaHat)
  
  # Argument 4 :: Phenotype names
  if(!missing(Phenotypes))
    checkPhen(Phenotypes, BetaHat)
  else Phenotypes = paste("trait", 1:length(BetaHat), sep = "")
  
  # Argument 5 :: Variant name
  if(!missing(Variant))
  {
    Variant <- checkVarName(Variant)
    variantName <- unname(Variant)           # Assignment
  }
  else variantName <- "Variant"
  
    # Argument 6 :: Update model parameter DE
    if(!is.logical(UpdateDE))
    {
      warning("UpdateDE not provided as logical (default option used).", call. = FALSE)
      UpdateDE <- TRUE
    }
    
    # Argument 7 :: Number of MCMC iteration
    mcmcDefault <- 20000
    # Check whether argument 7 is a vector of length 1
    if(!is.vector(MCMCiter) || (length(MCMCiter) != 1))
    {
      warning("MCMCiter is not a vector of length 1 (default option used).", call. = FALSE)
      MCMCiter <- mcmcDefault
    }
    # Check whether argument 7 is numeric and integer
    if(!is.numeric(MCMCiter) || MCMCiter%%1 != 0)
    {
      warning("MCMCiter not provided as integer (default option used).", call. = FALSE)
      MCMCiter <- mcmcDefault
    }
    # Check whether argument 7 is more than 10000
    if(MCMCiter < 10000)
    {
      warning("MCMCiter should be at least 10000 (default option used).", call. = FALSE)
      MCMCiter <- mcmcDefault
    }
    
    # Argument 8 :: Burnin
    BurninDefault <- 10000
    # Check whether argument 8 is a vector of length 1
    if(!is.vector(Burnin) || (length(Burnin) != 1))
    {
      warning("Burnin is not a vector of length 1 (default option used).", call. = FALSE)
      Burnin <- BurninDefault
    }
    # Check whether argument 8 is numeric and integer
    if(!is.numeric(Burnin) || Burnin%%1 != 0)
    {
      warning("Burnin not provided as integer (default option used).", call. = FALSE)
      Burnin <- BurninDefault
    }
    # Check whether argument 8 is more than 5000
    if(Burnin < 5000)
    {
      warning("Burnin should be at least 5000 (default option used).", call. = FALSE)
      Burnin <- BurninDefault
    }
    
    # Argument 7 and 8
    if((MCMCiter - Burnin) < 5000)
    {
      warning("MCMC sample size (MCMCiter - Burnin) provided less than 5000 (default options used)", call. = FALSE)
      MCMCiter <- mcmcDefault
      Burnin <- BurninDefault
    }
    
    # Call CPBayes function
    RESULT <- combined_CPBayes( variantName, Phenotypes, BetaHat, SE, COR, UpdateDE, MCMCiter, Burnin )
    print_result(RESULT)
    invisible(RESULT)
}



