% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/blc.R
\name{blc}
\alias{blc}
\title{Lee-Carter Bayesian Estimation for mortality data}
\usage{
blc(Y, prior = NULL, init = NULL, M = 2000, bn = 1000)
}
\arguments{
\item{Y}{Log-mortality rates for each age.}

\item{prior}{A list containing the prior mean \eqn{m_0} and the prior
variance \eqn{C_0}.}

\item{init}{A list containing initial values for \eqn{\alpha}, \eqn{\beta},
\eqn{\phi_V}, \eqn{\phi_W} and \eqn{\theta}.}

\item{M}{The number of iterations.}

\item{bn}{The number of initial iterations from the Gibbs sampler that should be discarded (burn-in). Default considers half the number of iterations.}
}
\value{
A \code{BLC} object.
\item{alpha}{Posterior sample from alpha parameter.}
\item{beta}{Posterior sample from beta parameter.}
\item{phiv}{Posterior sample from phiv parameter. phiv is the precision of the random error of the Lee Carter model.}
\item{theta}{Posterior sample from theta.}
\item{phiw}{Posterior sample from phiw. phiw is the precision of the random error of the random walk.}
\item{kappa}{Sampling from the state variables.}
\item{Y}{Y Log-mortality rates for each age passed by the user to fit the model.}
\item{bn}{The warmup of the algorithm specified by the user to fit the model.}
\item{M}{The number of iterations specified by the user to fit the model.}
\item{m0}{The prior mean of kappa0.}
\item{C0}{The prior covariance matrix of kappa0.}
}
\description{
Performs Bayesian estimation of the Lee-Carter model considering different
variances for each age.
}
\details{
Let \eqn{Y_{it}} be the log mortality rate at age \eqn{i} and time \eqn{t}. The Lee-Carter
model is specified as follows:

\eqn{Y_{it} = \alpha_i + \beta_i \kappa_t + \varepsilon_{it}, i=1,...,p} and \eqn{t=1,...,T},

where \eqn{\alpha = (\alpha_1,...,\alpha_p)'} are the interept of the model that represent
the log-mortality rate mean in each age; \eqn{\beta = (\beta_1,...,\beta_p)'} are the
coefficient regression that represent the speed of relative change in the log-mortality
rate in each age. \eqn{\kappa = (\kappa_1,...,\kappa_T)'} are the state variable that
represent the global relative change in log-mortality rate. Finally, \eqn{\varepsilon_{it} ~ N(0, \sigma^2_i)}
is the random error.

For the state variable \eqn{\kappa_t} Lee and Carter (1992) proposed a random walk with
drift to govern the dynamics over time:

\eqn{\kappa_t = \kappa_{t-1} + \theta + \omega_t},

where \eqn{\theta} is the drift parameter and \eqn{\omega_t} is the random error of the
random walk.

We implement the Bayesian Lee Carter (BLC) model, proposed by Pedroza (2006), to estimate
the model. In this approach, we take advantage of the fact that the Bayesian Lee Carter
can be specified as dynamic linear model, to estimate the state variables \eqn{\kappa_t}
through FFBS algorithm. To estimate the others parameters we use Gibbs sampler to sample
from their respective posterior distribution.
}
\examples{
## Example of transforming the dataset to fit the function:

## Importing mortality data from the USA available on the Human Mortality Database (HMD):
data(USA)

## Calculating the mortality rates for the general population:
require(dplyr)
require(tidyr)
require(magrittr)

USA \%>\% mutate(mx = USA$Dx.Total/USA$Ex.Total) -> data

data \%>\%
	filter(Age \%in\% 18:80 & Year \%in\% 2000:2015)  \%>\%
	mutate(logmx = log(mx)) \%>\%
	dplyr::select(Year,Age,logmx) \%>\%
	pivot_wider(names_from = Year, values_from = logmx) \%>\%
	dplyr::select(-Age) \%>\%
	as.matrix()  \%>\%
	magrittr::set_rownames(18:80) -> Y

## Fitting the model
fit = blc(Y = Y, M = 100, bn = 20)
print(fit)

## Viewing the results
plot(fit, ages = 18:80)
plot(fit, parameter = "beta", ages=18:80)
improvement(fit)

}
\references{
Lee, R. D., & Carter, L. R. (1992). “Modeling and forecasting US mortality.” \emph{Journal of the American statistical association}, 87(419), 659-671.

Pedroza, C. (2006). “A Bayesian forecasting model: predicting US male mortality.” \emph{Biostatistics}, 7(4), 530-550.
}
\seealso{
\code{\link[=fitted.BLC]{fitted.BLC()}}, \code{\link[=plot.BLC]{plot.BLC()}}, \code{\link[=print.BLC]{print.BLC()}} and \code{\link[=predict.BLC]{predict.BLC()}} for \code{BLC} methods to native R functions \code{\link[=fitted]{fitted()}},
\code{\link[=plot]{plot()}}, \code{\link[=print]{print()}} and \code{\link[=predict]{predict()}}.

\code{\link[=expectancy.BLC]{expectancy.BLC()}} and \code{\link[=Heatmap.BLC]{Heatmap.BLC()}} for \code{BLC} methods to compute and visualise the truncated life expectancy
via \code{\link[=expectancy]{expectancy()}} and \code{\link[=Heatmap]{Heatmap()}} functions.

\code{\link[=improvement]{improvement()}} to compute the improvement of each age, based on the resulting chains of the beta parameter.
}
