% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bitmapfont.r
\name{rasterToAmigaBitmapFont}
\alias{rasterToAmigaBitmapFont}
\title{Convert a raster image into an AmigaBitmapFont}
\usage{
rasterToAmigaBitmapFont(x, glyphs, default_glyph, baseline, glyph_width,
  glyph_space, glyph_kern, palette, ...)
}
\arguments{
\item{x}{A \code{raster} (see grDevices package) object composed of
two colours only. Make sure that all glyphs (graphical representation
of characters) are next to eachother on a single line. The height
of this raster (in pixels) is taken automatically as font height.}

\item{glyphs}{Specify which glyphs are included in the image
\code{x} from left to right. It can be specified in one of the
following ways:

A single \code{character} string, where the length of the string
(\code{nchar}) equals the number of displayed glyphs in \code{x}.

A \code{vector} of \code{numeric} ASCII codes. The length of
the \code{vector} should equal the number of displayed glyphs
in \code{x}.

A \code{list} of either \code{character} strings or \code{vector} of
\code{numeric}s.  The length of the \code{list} should equal the
number of displayed glyphs in \code{x}. Each element can represent
multiple characters, meaning that the nth element of the list
uses the nth glyph shown in \code{x} to represent all the characters
included in that element.

Note that Amiga bitmap fonts represent ASCII characters and may
not include all special characters or symbols.}

\item{default_glyph}{A single \code{character} or ASCII code
(\code{numeric}) that should be used by default. This means
that all characters that are not specified by \code{glyphs} will
be represented by this \code{default_glyph}. \code{default_glyph} should
be included in \code{glyphs}.}

\item{baseline}{The baseline of the font, specified in number of
pixels from the top (\code{numeric}). Should be a whole number
between 0 and the font height (height of \code{x}) minus 1.}

\item{glyph_width}{A \code{numeric} \code{vector} with the same number
of elements or characters as used for \code{glyphs}. It specifies
the width in pixels for each glyph reserved in the raster image \code{x}.
They should be whole numbers greater or equal to 0.}

\item{glyph_space}{A \code{numeric} \code{vector} with the same number
of elements or characters as used for \code{glyphs}. It specifies
the width in pixels for each glyph that should be used when formatting.
text. Note that these values can be smaller or larger than the values
specified for \code{glyph_width}.
They should be whole numbers greater or equal to 0.}

\item{glyph_kern}{Not that in Amiga bitmap fonts not the formal
definition from typography is used for kerning. Here, kerning is
used as the number of pixels the cursor should be moved forward or
backward after typesetting a character. It should be a
\code{numeric} \code{vector} with the same number of elements or
characters as used for \code{glyphs}. It can hold both positive
and negative values.}

\item{palette}{A \code{vector} of two colours. Both colours should
be in \code{x}. The first colour is used as background colour,
the second as foreground colour.

When missing, it will be checked whether \code{x} has a palette
as attribute, and uses that. If that attribute is also missing,
the palette will be guessed from \code{x}, where the most
frequently occuring colour is assumed to be the background
colour.}

\item{...}{Currently ignored.}
}
\value{
Returns a \code{\link{AmigaBitmapFont}} class object based on \code{x}.
}
\description{
Convert a two-coloured \code{\link[grDevices]{as.raster}} image into
an \code{\link{AmigaBitmapFont}} class object.
}
\details{
Create an \code{\link{AmigaBitmapFont}} class object by providing
a two-coloured raster image and specifying which characters are
depicted by the image.
}
\examples{
\dontrun{
data("font_example")

## make a raster that we can use to create a bitmap font
font9.rast <- as.raster(getAmigaBitmapFont(font_example, 9))

## note the glyphs and the order in which they are included in
## the raster image:
plot(font9.rast)

## let's build a simple font, using only the first few glyphs
## in the raster:
font9 <- rasterToAmigaBitmapFont(
  ## 'x' needs the raster image:
  x             = font9.rast,
  
  ## 'glyphs' are the graphical representation of the characters
  ## that we will include in our font. We will only use the
  ## first 7 characters in the raster image:
  glyphs        = " !\\"#$\%&",
  
  ## We will use the '&' glyph to represent all characters that
  ## are not specified in the font:
  default_glyph = "&",
  
  ## The raster image is 9 pixels tall, as will be the font.
  ## Let's use 7 as the base (it needs to be less than the height)
  baseline      = 7,
  
  ## Let's define the width in pixels for each of the 7
  ## characters. This is their width in the raster image:
  glyph_width   = c(0, 1, 3, 6, 5, 5, 5),
  
  ## Let's define the space the character should take in pixels
  ## when it is used to format text:
  glyph_space   = c(4, 2, 4, 7, 6, 6, 6),
  
  ## the raster uses white as background colour and black as
  ## foreground:
  palette       = c("white", "black")
)

## note that for all characters that are not specified,
## the default glyph ('&') is used:
plot(font9, text = "!@#$\%ABCD")

## Let's take a subset from the font's bitmap (rasteer):
font9abc.rast <- font9.rast[,263:282]

## as you can see this bitmap only contains the lowercase
## characters 'a', 'b', 'c', 'd' and 'e':
plot(font9abc.rast)

font9.abc <- rasterToAmigaBitmapFont(
  x             = font9abc.rast,
  ## Each glyph in the image can be represented by a single
  ## element in a list. By specifying multiple characters in
  ## each element, you can recycle a glyph to represent different
  ## characters. So in this case, the glyph 'a' is used for
  ## all the accented variants of the character 'a'.
  glyphs        = list("a\\xE0\\xE1\\xE2\\xE3\\xE4\\xE5",
                       "b",
                       "c\\xA2\\xE7",
                       "d",
                       "e\\xE8\\xE9\\xEA\\xEB"),
  default_glyph = "c", ## 'c' is used as default glyph for all other characters
  baseline      = 7,
  glyph_width   = c(4, 4, 4, 4, 4),
  glyph_space   = c(5, 5, 5, 5, 5),
  palette       = c("white", "black")
)

## see what happens when you format text using the font we just created:
plot(font9.abc, text = "a\\xE0\\xE1\\xE2\\xE3\\xE4\\xE5bc\\xA2\\xE7de\\xE8\\xE9\\xEA\\xEB, foo bar")
}
}
\seealso{
Other AmigaBitmapFont.operations: \code{\link{AmigaBitmapFont}},
  \code{\link{availableFontSizes}}, \code{\link{c}},
  \code{\link{fontName}}, \code{\link{font_example}},
  \code{\link{getAmigaBitmapFont}},
  \code{\link{rawToAmigaBitmapFontSet}},
  \code{\link{rawToAmigaBitmapFont}},
  \code{\link{read.AmigaBitmapFontSet}},
  \code{\link{read.AmigaBitmapFont}},
  \code{\link{write.AmigaBitmapFont}}

Other raster.operations: \code{\link{AmigaBitmapFont}},
  \code{\link{as.raster}}, \code{\link{bitmapToRaster}},
  \code{\link{dither}}, \code{\link{index.colours}},
  \code{\link{rasterToBitmap}},
  \code{\link{rasterToHWSprite}}, \code{\link{rasterToIFF}}
}
\author{
Pepijn de Vries
}
