% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/first_isolates.R
\name{first_isolate}
\alias{first_isolate}
\title{Determine first (weighted) isolates}
\source{
Methodology of this function is based on: "M39 Analysis and Presentation of Cumulative Antimicrobial Susceptibility Test Data, 4th Edition", 2014, Clinical and Laboratory Standards Institute. \url{https://clsi.org/standards/products/microbiology/documents/m39/}.
}
\usage{
first_isolate(tbl, col_date, col_patient_id, col_bactid = NA,
  col_testcode = NA, col_specimen = NA, col_icu = NA,
  col_keyantibiotics = NA, episode_days = 365, testcodes_exclude = "",
  icu_exclude = FALSE, filter_specimen = NA, output_logical = TRUE,
  type = "keyantibiotics", ignore_I = TRUE, points_threshold = 2,
  info = TRUE, col_genus = NA, col_species = NA)
}
\arguments{
\item{tbl}{a \code{data.frame} containing isolates.}

\item{col_date}{column name of the result date (or date that is was received on the lab)}

\item{col_patient_id}{column name of the unique IDs of the patients}

\item{col_bactid}{column name of the unique IDs of the microorganisms (should occur in the \code{\link{microorganisms}} dataset)}

\item{col_testcode}{column name of the test codes. Use \code{col_testcode = NA} to \strong{not} exclude certain test codes (like test codes for screening). In that case \code{testcodes_exclude} will be ignored. Supports tidyverse-like quotation.}

\item{col_specimen}{column name of the specimen type or group}

\item{col_icu}{column name of the logicals (\code{TRUE}/\code{FALSE}) whether a ward or department is an Intensive Care Unit (ICU)}

\item{col_keyantibiotics}{column name of the key antibiotics to determine first \emph{weighted} isolates, see \code{\link{key_antibiotics}}. Supports tidyverse-like quotation.}

\item{episode_days}{episode in days after which a genus/species combination will be determined as 'first isolate' again}

\item{testcodes_exclude}{character vector with test codes that should be excluded (case-insensitive)}

\item{icu_exclude}{logical whether ICU isolates should be excluded}

\item{filter_specimen}{specimen group or type that should be excluded}

\item{output_logical}{return output as \code{logical} (will else be the values \code{0} or \code{1})}

\item{type}{type to determine weighed isolates; can be \code{"keyantibiotics"} or \code{"points"}, see Details}

\item{ignore_I}{logical to determine whether antibiotic interpretations with \code{"I"} will be ignored when \code{type = "keyantibiotics"}, see Details}

\item{points_threshold}{points until the comparison of key antibiotics will lead to inclusion of an isolate when \code{type = "points"}, see Details}

\item{info}{print progress}

\item{col_genus}{(deprecated, use \code{col_bactid} instead) column name of the genus of the microorganisms}

\item{col_species}{(deprecated, use \code{col_bactid} instead) column name of the species of the microorganisms}
}
\value{
A vector to add to table, see Examples.
}
\description{
Determine first (weighted) isolates of all microorganisms of every patient per episode and (if needed) per specimen type.
}
\details{
\strong{WHY THIS IS SO IMPORTANT} \cr
    To conduct an analysis of antimicrobial resistance, you should only include the first isolate of every patient per episode \href{https://www.ncbi.nlm.nih.gov/pubmed/17304462}{[1]}. If you would not do this, you could easily get an overestimate or underestimate of the resistance of an antibiotic. Imagine that a patient was admitted with an MRSA and that it was found in 5 different blood cultures the following week. The resistance percentage of oxacillin of all \emph{S. aureus} isolates would be overestimated, because you included this MRSA more than once. It would be \href{https://en.wikipedia.org/wiki/Selection_bias}{selection bias}.

    \strong{DETERMINING WEIGHTED ISOLATES} \cr
    \strong{1. Using} \code{type = "keyantibiotics"} \strong{and parameter} \code{ignore_I} \cr
    To determine weighted isolates, the difference between key antibiotics will be checked. Any difference from S to R (or vice versa) will (re)select an isolate as a first weighted isolate. With \code{ignore_I = FALSE}, also differences from I to S|R (or vice versa) will lead to this. This is a reliable method and 30-35 times faster than method 2. \cr
    \strong{2. Using} \code{type = "points"} \strong{and parameter} \code{points_threshold} \cr
    To determine weighted isolates, difference between antimicrobial interpretations will be measured with points. A difference from I to S|R (or vice versa) means 0.5 points, a difference from S to R (or vice versa) means 1 point. When the sum of points exceeds \code{points_threshold}, an isolate will be (re)selected as a first weighted isolate. This method is being used by the Infection Prevention department (Dr M. Lokate) of the University Medical Center Groningen (UMCG).
}
\examples{
# septic_patients is a dataset available in the AMR package
?septic_patients
my_patients <- septic_patients

library(dplyr)
my_patients$first_isolate <- my_patients \%>\%
  first_isolate(col_date = "date",
                col_patient_id = "patient_id",
                col_bactid = "bactid")

\dontrun{

# set key antibiotics to a new variable
tbl$keyab <- key_antibiotics(tbl)

tbl$first_isolate <-
  first_isolate(tbl)

tbl$first_isolate_weighed <-
  first_isolate(tbl,
                col_keyantibiotics = 'keyab')

tbl$first_blood_isolate <-
  first_isolate(tbl,
                filter_specimen = 'Blood')

tbl$first_blood_isolate_weighed <-
  first_isolate(tbl,
                filter_specimen = 'Blood',
                col_keyantibiotics = 'keyab')

tbl$first_urine_isolate <-
  first_isolate(tbl,
                filter_specimen = 'Urine')

tbl$first_urine_isolate_weighed <-
  first_isolate(tbl,
                filter_specimen = 'Urine',
                col_keyantibiotics = 'keyab')

tbl$first_resp_isolate <-
  first_isolate(tbl,
                filter_specimen = 'Respiratory')

tbl$first_resp_isolate_weighed <-
  first_isolate(tbl,
                filter_specimen = 'Respiratory',
                col_keyantibiotics = 'keyab')
}
}
\keyword{first}
\keyword{isolate}
\keyword{isolates}
