aero_calc <-function(df, sensor_height, timestep, out_file_format,
                     out_file_name, verbose = TRUE){
  #' Estimates open water evaporation using the aerodynamic mass transfer
  #' approach for the entire analysis period
  #'
  #' This function takes a dataframe input data and calculates evaporation from
  #' input parameters
  #'
  #' @param df Dataframe containing input data. Must include all data components
  #'          including 'T_skin', 'WS', 'P', 'T_air', 'RH', 'date'. Ensure date is formatted as a date object before reading into function.
  #' @param sensor_height sensor height (m)
  #' @param timestep measurement frequency (s). For example, hourly data would be timestep = 3600
  #' @param out_file_format "csv" or "rds"
  #' @param out_file_name name to save output file to
  #' @param verbose logical indicator to suppress messages
  #'
  #' @return list with calculated values for evaporation (mm/timestep), bulk
  #' transfer coefficient, vapor pressure deficit (kPa), and most stability value
  #'


  # Convert Pressure units
  df$P <- as.numeric(df$P)

  #Add columns to df
  df$SH <- sensor_height
  df$dt <- timestep

  #Set columns in df to numeric
  numeric_vars = c('WS', 'P', 'T_air', 'T_skin', 'RH', 'SH')
  df[numeric_vars] <- lapply(df[numeric_vars], as.numeric)

  #Add additional columns for results to df
  df$E <- 0
  df$Ce <- 0
  df$VPD <- 0
  df$stability <- 0


  for(i in 1:nrow(df)) {
    row <- df[i,]

    new_var_names <- c('T_skin', 'WS', 'P', 'T_air', 'RH', 'date')
    check <- c(row['WS'], row['P'], row['T_air'], row['T_skin'], row['RH'], row['SH'])
    if (any(is.nan(unlist(check, use.names=FALSE)))){
      message('One or more variables missing')
      return(c(NaN, NaN, NaN, NaN))
    }

    datetime = row[['date']]
    wind = row[['WS']]
    pressure = row[['P']]
    T_air = row[['T_air']]
    T_skin = row[['T_skin']]
    RH = row[['RH']]
    sensor_height = row[['SH']]
    timestep = row[['dt']]


    #Constants
    K=0.41 #von Karman constant
    g=9.81 #gravity (m/s^2)
    a=0.0123 #Charnock constant

    ############################################################
    #Calculate meterological variables
    #Sensort height (m)
    z=sensor_height

    #Convert from Celcius to Kelvin
    T_air=T_air+273.15
    T_skin=T_skin+273.15

    #Potential temperatures (air and skin) Kelvin
    T_air_pot=(T_air)*(1000./pressure)**(0.286)
    T_skin_pot=(T_skin)*(1000./pressure)**(0.286)

    #Atmospheric vapor pressure (kPa) (2m)
    e_air=(RH/100)*(0.6108*exp(((17.27*(T_air-273.15))/((T_air-273.15)+237.3))))

    #Atmospheric specific humidity (kg/kg) (2m)
    q_air=0.62*e_air/(pressure/10-0.38*e_air)

    #Saturated Water-Surface vapor pressure (kPa) (0m)
    e_sat=0.6108*exp(((17.27*(T_skin-273.15))/((T_skin-273.15)+237.3)))

    #Saturated specific humidity at water surface (kg/kg)  (0m)
    q_sat=0.62*e_sat/(pressure/10-0.38*e_sat)

    #Vapor Pressure Deficit (kPa)
    VPD=e_sat-e_air

    #Density of air (kg/m^3)
    density_air=(pressure/10*1000)/((T_air)*286.9*(1+0.61*q_air))

    #Kinematic viscocity
    #Estimated using data from Montgomery, 1947 in Verburg, 2010
    v=(4.94*10**-8*(T_air-273.15)+1.7185*10**-5)/density_air

    #Virtual Temperature
    Tv=T_air*(1+q_air*0.61)

    ###############################################################
    # Bulk Transfer Coefficient Iteration, Ce

    #Stable Condition (z/L > 0)
    #Initial Values for Iteration
    #Stability Function (Momentum)
    Sm=0
    #Stability Function (Temperature)
    St=0
    #Stability Function (Vapor)
    Sq=0
    #Friction velocity
    Us=0
    #Roughness Length of Momentum
    zo=.00010

    #Friction Velocity of Momentum
    u_f=(K*(wind-Us))/(log(z/zo)-Sm)
    #Roughness Length of Vapor
    zoq=7.4*zo*exp(-2.25*(zo*u_f/v)**.25)
    #Roughness Legnth of Temperature
    zot=7.4*zo*exp(-2.25*(zo*u_f/v)**.25)
    #Scaling Potential Temperature
    t_fv=(K*(T_air_pot-T_skin_pot))/(log(z/zot)-St)

    #Scaling Humidity
    q_f=(K*(q_air-q_sat))/(log(z/zoq)-Sq)
    #Monin-Obhukov Length
    L=(Tv*u_f**2)/(K*g*t_fv)

    for (x in 0:198){
      #Friction Velocity of Momentum
      u_f=(K*(wind-u_f))/(log(z/zo)-Sm)
      if (is.nan(u_f[[1]])){
        if (verbose){
        message("Invalid value encountered and prevented stable condition convergence for ", toString(datetime))
        }
        break
      }
      #Scaling Potential Temperature
      t_fv=(K*(T_air_pot-T_skin_pot))/(log(z/zot)-St)
      #Scaling Humidity
      q_f=(K*(q_air-q_sat))/(log(z/zoq)-Sq)
      #Stability Function of Momentum
      Sm=as.double(-5.2*(z))/L
      #Stability Function of Vapor
      Sq=as.double(-5.2*(z))/L
      #Roughness Length of Momentum
      zc=a*u_f**2/g
      zs=0.11*v/u_f
      zo=zc+zs
      #Roughness Length of Vapor
      zoq=7.4*zo*exp(-2.25*(zo*u_f/v)**.25)
      #Monin-Obhukov Length
      L=(Tv*u_f**2)/(K*g*t_fv)
      if (is.infinite(L)){
        L = NA
      }
      if (is.nan(L[[1]])){
        if (verbose){
        message("Invalid value encountered and prevented stable condition convergence")
        }
        break
      }
    }

    stability_s = z/L
    if (is.nan(stability_s) || is.na(stability_s)){
      if (verbose){
      message("Invalid value encountered and prevented stable condition convergence")
      }
      stability_s = 0
    }

    if (is.complex(L)){
      Ce_s= NaN
    }else{
      if (is.complex(stability_s)){
        message("check")
        if (Re(stability_s) > 0){
          Ce_s=Re(K**2/((log(z/zo)-Sm)*(log(z/zoq)-Sq)))
        }else{
          Ce_s=NaN
        }
      }else{
        if (stability_s > 0){
          Ce_s=K**2/((log(z/zo)-Sm)*(log(z/zoq)-Sq))
        }else{
          Ce_s=NaN
        }
      }
    }
    #########################################################################
    #Unstable Conditions (z/L< 0)
    #Initial Values for Iteration
    #Stability Function (Momentum)
    Sm=0
    #Stability Function (Temperature)
    St=0
    #Stability Function (Vapor)
    Sq=0
    #Friction velocity
    Us=0
    #Roughness Length of Momentem
    zo=.00010

    #Friction Velocity of Momentum
    u_f=(K*(wind-Us))/(log(z/zo)-Sm);
    #Roughness Length of Vapor
    zoq=7.4*zo*exp(-2.25*(zo*u_f/v)**.25);
    #Roughness Legnth of Temperature
    zot=7.4*zo*exp(-2.25*(zo*u_f/v)**.25);
    #Scaling Potential Temperature
    t_fv=(K*(T_air_pot-T_skin_pot))/((z/zot)-St);
    #Scaling Humidity
    q_f=(K*(q_air-q_sat))/(log(z/zoq)-Sq);
    #Monin-Obhukov Length
    L=(Tv*u_f**2)/(K*g*t_fv)

    for (x in 0:198){
      #Friction Velocity of Momentum
      u_f=(K*(wind-u_f))/(log(z/zo)-Sm)
      if (is.nan(u_f[[1]])){
        if (verbose){
        message("Invalid value encountered and prevented unstable condition convergence for ", toString(datetime))
        }
        break
      }

      #Scaling Temperature
      t_fv=(K*(T_air_pot-T_skin_pot))/(log(z/zot)-St)
      #Scaling Humidity
      q_f=(K*(q_air-q_sat))/(log(z/zoq)-Sq)
      #Input for Stability function calculations
      x=(1-16*(z/L))**.25
      #Stability Function of Momentum
      Sm=2*log((1+x)/2)+log((1+x**2)/2)-2*atan(x)+(pi/2)
      #Stability Function of Vapor
      Sq=2*log((1+x**2)/2)
      #Roughness Length of Momemtum
      zc=a*u_f**2/g
      zs=0.11*v/u_f
      zo=zc+zs
      #Roughness Length of Vapor
      zoq=7.4*zo*exp(-2.25*(zo*u_f/v)**.25)
      #Monin-Obhukov Length
      L=(Tv*u_f**2)/(K*g*t_fv)
      if (is.infinite(L)){
        L = NA
      }
      if (is.nan(L[[1]])){
        if (verbose){
        message("Invalid value encountered and prevented unstable condition convergence")
        }
        break
      }
    }

    stability_u = z/L

    if (is.nan(stability_u)){
      if (verbose){
      message("Invalid value encountered and prevented stable condition convergence")
      }
      stability_u = 0
    }

    if (is.complex(L)){
      Ce_u=NaN
    }else{
      if (is.complex(stability_u)){
        if (Re(stability_u) > 0){
          Ce_u=Re(K**2/((log(z/zo)-Sm)*(log(z/zoq)-Sq)))
        }else{
          Ce_u=NaN
        }
      }else{
        Ce_u=K**2/((log(z/zo)-Sm)*(log(z/zoq)-Sq))
      }
    }

    #################################################################
    #Neutral Conditions, z/L=0
    #Initial Conditions
    zo=.00010

    for (x in 0:198){
      #Friction Velocity of Momentum
      u_f=(K*wind)/(log(z/zo))
      if (is.nan(u_f[[1]])){
        if (verbose){
        message("Invalid value encountered and prevented neutral condition convergence for ", toString(datetime))
        }
        break
      }
      #Roughness Length of Momemtum
      zc=(a*u_f**2)/g
      zs=0.11*v/u_f
      zo=zc+zs
      #Roughness Length of Vapor
      zoq=7.4*zo*exp(-2.25*(zo*u_f/v)**.25)
    }

    Ce_n=(K**2)/((log(z/zo))*(log(z/zoq)))

    ################################################################
    #Assign correct Ce value (stable, unstable, or neutral)
    if (is.finite(Ce_s[[1]])){
      Ce=Ce_s
      stability = stability_s
    }else{
      if (is.finite(Ce_u[[1]])){
        Ce=Ce_u
        stability = stability_u
      }else{
        Ce=Ce_n
        stability = 0
      }
    }
    ################################################################
    #Calculated evaporation in mm/timestep
    E=density_air*Ce*(q_sat-q_air)*wind*timestep

    df[i, "E"] <- E
    df[i, "Ce"] <- Ce
    df[i, "VPD"] <- VPD
    df[i, "stability"] <- stability
  }

  if (out_file_format == "csv") {
    write.csv(df, paste0(out_file_name, ".csv"))
  } else if (out_file_format == "rds") {
    saveRDS(df, file = paste0(out_file_name, ".rds"))
  }

  #Add columns to put the results back into the df
  return(df)

}
