% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/inspire.R
\name{inspire}
\alias{inspire}
\alias{z22_inspire_generate}
\alias{z22_inspire_extract}
\title{Generate INSPIRE IDs}
\usage{
z22_inspire_generate(coords, res = NULL, short = FALSE, llc = FALSE)

z22_inspire_extract(inspire, as = c("df", "sf"), meta = FALSE)
}
\arguments{
\item{coords}{A list, matrix, or dataframe where the X and Y coordinates are
either in the columns \code{"x"} and \code{"y"} or in the first and second
column position, respectively. Column names are converted to lowercase.

Can also be a \code{sf}/\code{sfc} object in which case the coordinates are
extracted using \code{\link[sf]{st_coordinates}}.}

\item{res}{Resolution of the grid. Can be \code{"100m"}, \code{"250m"},
\code{"1km"}, \code{"5km"}, \code{"10km"}, or \code{"100km"}. If
\code{NULL}, tries to guess the resolution from the provided coordinates.}

\item{short}{If \code{TRUE}, generates short INSPIRE IDs. Defaults to
\code{FALSE}.}

\item{llc}{Do the coordinates in \code{coords} represent the lower-left
corners of their cells? If \code{FALSE}, subtracts each coordinate by
half of \code{res}. If \code{TRUE}, leaves them as-is. Defaults
to \code{FALSE}, i.e., treat coordinates as cell centroids.}

\item{inspire}{A vector of INSPIRE IDs. Can be either short or long format.}

\item{as}{Specifies the output class. Must be one of \code{"df"} or
\code{"sf"}. If \code{"df"} (default), returns flat
coordinates in a dataframe. If \code{"sf"} (and the \code{sf} package is
installed), converts the coordinates to an \code{sf} tibble.}

\item{meta}{Whether to include parsed CRS and resolution in the output.
If \code{FALSE}, output contains only coordinates. If \code{TRUE}, also
contains columns \code{"crs"} and \code{"res"}.}
}
\value{
\code{z22_inspire_generate} returns a character vector containing
the INSPIRE identifiers. \code{z22_inspire_extract} returns a dataframe
or \code{\link[sf:st_sfc]{sfc}} object containing the points extracted from
the INSPIRE identifiers. Note that the returned coordinates are always
the centers of the grid cells as opposed to the south-west corners.
}
\description{
Given pairs of coordinates, generates their INSPIRE grid representation.
Given INSPIRE identifiers, can also extract the X and Y coordinates.

An INSPIRE ID contains information about the CRS, cell size and the
ETRS89-LAEA coordinates of the south-west corner of the grid cell in its
format.

\preformatted{CRS3035RES{cellsize}mN{y}E{x} # long format
{cellsize}N{y}E{x}         # short format}

The short format always uses meters while the short formats aggregates
cell sizes greater or equal to 1000m to km.
}
\details{
To remain fast even for huge grid datasets, the function is just a very
simple \code{\link{sprintf}} wrapper that performs no input checks. To
produce valid INSPIRE identifiers, make sure to transform your data to
ETRS89-LAEA (e.g. using
\code{\link[sf:st_transform]{st_transform}(..., 3035)}). You should also
make sure that the coordinates are the south-west corner of existing
INSPIRE grid cells.
}
\examples{
library(dplyr, warn.conflicts = FALSE)

# Generate IDs from a dataframe
coords <- tibble(x = c(4334150, 4334250), y = c(2684050, 2684050))
identical(z22_inspire_extract(z22_inspire_generate(coords))[c("x", "y")], coords)

# Extract coordinates from short ID strings
z22_inspire_extract("100mN34000E44000")

# Generate IDs from an sf dataframe
if (requireNamespace("sf", quietly = TRUE)) {
  coords <- sf::st_as_sf(coords, coords = c("x", "y"), crs = 3035)
  z22_inspire_generate(coords)
}
}
